/**
 * Editable details view reuses components from JIRA platform such as the comments section and attachments section.
 * These components have links which activate dialogs, such as deleting an attachment or editing a comment.
 * By default, submitting one of these dialogs redirects to the view issue page. We don't want to do this,
 * so we need to intercept the Dialog.show call and hook in a custom onDialogFinished.
 * The reason why we cant do this the old fashioned way (e.g. gh-InitDialogs.js) is that we do not call these dialogs,
 * platform does, so we have no other way to supply a submit callback.
 */
define("jira-agile/rapid/ui/detail/inlineedit/dialogsoverrider", ["require"], function (require) {
    "use strict";

    var Events = require("jira/util/events");
    var jQuery = require('jquery');
    var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
    var DetailsModel = GH.DetailsObjectFactory.getDetailsModel();

    var DialogsOverrider = {};

    DialogsOverrider.DIALOG_OVERRIDE_LIST = ['delete-attachment-dialog', 'delete-issue-link-dialog', 'edit-comment',
                                             'delete-comment-dialog', 'assign-to-me-link-handler', 'assign-dialog',
                                             'link-issue-dialog'];

    var onDialogShow = null;

    function shouldOverrideDialogWithId(id) {
        return DialogsOverrider.DIALOG_OVERRIDE_LIST.indexOf(id) !== -1;
    }

    DialogsOverrider.init = function (dialogSubmitHandler) {
        if (typeof onDialogShow === 'function') {
            return;
        }

        onDialogShow = function (event, $el, dialog) {
            var shouldOverrideDialog = dialog &&
                shouldOverrideDialogWithId(dialog.options.id);

            if (!shouldOverrideDialog) {
                return;
            }

            dialog._performRedirect = function () {
                this.redirected = false;
            };

            dialog.options.onDialogFinished = function () {
                dialogSubmitHandler(null, EditableDetailsViewReloadReason.REASON_FOR_DIALOG[dialog.options.id]);
                AJS.$(GH).trigger('issueUpdated', {issueId: DetailsModel.viewData.id, source: 'detailView'});
                dialog.hide();
            };
        };

        Events.bind("Dialog.show", onDialogShow);

        Events.bind("Dialog.hide", function(event, $el, dialog) {
            // JIRA dialogs add in an element with ".aui-blanket" but do not remove it after they are closed.
            // Remove ".aui-blanket" so other plugins (eg. jira-issue-navigator-components-plugin) do not block
            // editable fields in ADV from getting focused after entering edit mode
            jQuery('.aui-blanket').remove();
        });
    };

    DialogsOverrider.destroy = function () {
        Events.unbind("Dialog.show", onDialogShow);
        onDialogShow = null;
    };

    return DialogsOverrider;
});
