define('jira-agile/rapid/ui/detail/inlineedit/timetracking-fields-helper', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var IssueFieldUtil = require('jira/components/issueviewer/legacy/issuefieldutil');

    var ORIGINAL_TIME_ESTIMATE = 'timeoriginalestimate';
    var REMAINING_TIME_ESTIMATE = 'timeestimate';
    var TARGET_FIELD_PARAM_NAME = 'timetracking_targetsubfield';

    var TIME_TRACKING_FIELD_IDS = [ORIGINAL_TIME_ESTIMATE, REMAINING_TIME_ESTIMATE];

    var SERVER_EXPECTED_SAVED_FIELD_ID = 'timetracking';

    // Element ID for the form parameter which will store the value to save.
    var ORIGINAL_ESTIMATE_FORM_PARAM_ID = 'timetracking_originalestimate';
    var REMAINING_ESTIMATE_FORM_PARAM_ID = 'timetracking_remainingestimate';

    var TimeTrackingFieldsHelper = {
        _visibleForTesting: {
            ORIGINAL_TIME_ESTIMATE: ORIGINAL_TIME_ESTIMATE,
            REMAINING_TIME_ESTIMATE: REMAINING_TIME_ESTIMATE,
            TARGET_FIELD_PARAM_NAME: TARGET_FIELD_PARAM_NAME,
            ORIGINAL_ESTIMATE_FORM_PARAM_ID: ORIGINAL_ESTIMATE_FORM_PARAM_ID,
            REMAINING_ESTIMATE_FORM_PARAM_ID: REMAINING_ESTIMATE_FORM_PARAM_ID
        }
    };

    var savedOnlyOneField;

    /**
     * Fetches the time tracking input field from the given container.
     * @param fieldId the timetracking field which should be extracted.
     * @param $container edit form for the time tracking field.
     * @returns the jQuery object wrapping the input field
     *  or undefined if a non-timetracking field was requested.
     */
    TimeTrackingFieldsHelper.getFieldFromContainer = function (fieldId, $container) {
        if (this.isTimeTrackingField(fieldId)) {
            return $container.find('#' + getFormParam(fieldId));
        }
    };

    /**
     * If we want to save a time tracking field, we must tell the issue action that we are only saving one component
     * of the tracking field (by default it will attempt to save both). This method modifies the form data in place
     * in this case.
     * @param field the field to save. If it is not a time tracking field this method will not do anything.
     * @param data the form data that will be modified.
     * @param currently editing fields
     */
    TimeTrackingFieldsHelper.specifyWhichFieldToSave = function (field, data, editsInProgress) {
        var $thisTrackingField = $(IssueFieldUtil.getFieldSelector(field));
        var $otherTrackingField = $(IssueFieldUtil.getFieldSelector(getOther(field)));
        var otherTrackingFieldIsEmpty;

        if ($otherTrackingField.length !== 0) {
            // .ghx-fa is the canonical class used to style empty values. So we'll look for that.
            otherTrackingFieldIsEmpty = $otherTrackingField.find('.ghx-fa').length !== 0;
        } else {
            // Not using both tracking fields, so instead we'll look up the value from the form.
            var otherInput = this.getFieldFromContainer(getOther(field), $thisTrackingField);
            otherTrackingFieldIsEmpty = _.isEmpty(otherInput.val());
        }

        if (otherTrackingFieldIsEmpty && !_.contains(editsInProgress, getOther(field))) {
            // Save both fields
            delete data[getFormParam(getOther(field))];
        } else {
            // Save single field
            data[TARGET_FIELD_PARAM_NAME] = getFormParam(field);
            savedOnlyOneField = true;
        }
    };

    /**
     * If the saved field is one of two tracking fields, remove the other from the fields array and return the modified array.
     * @param savedField the field that was saved
     * @param fields the array of fields to remove from
     * @returns the modified array
     */
    TimeTrackingFieldsHelper.removeOtherTrackingField = function (savedField, fields) {
        if (savedOnlyOneField) {
            return _.reject(fields, function (f) {
                return TimeTrackingFieldsHelper.isTimeTrackingField(savedField) && f.id === getOther(savedField);
            });
        } else {
            return fields;
        }
    };

    TimeTrackingFieldsHelper.getServerFieldId = function () {
        return SERVER_EXPECTED_SAVED_FIELD_ID;
    };

    TimeTrackingFieldsHelper.isTimeTrackingField = function (field) {
        return _.contains(TIME_TRACKING_FIELD_IDS, field);
    };

    function getOther(fieldId) {
        if (fieldId === REMAINING_TIME_ESTIMATE) {
            return ORIGINAL_TIME_ESTIMATE;
        } else if (fieldId === ORIGINAL_TIME_ESTIMATE) {
            return REMAINING_TIME_ESTIMATE;
        }
    }

    function getFormParam(fieldId) {
        if (fieldId === REMAINING_TIME_ESTIMATE) {
            return REMAINING_ESTIMATE_FORM_PARAM_ID;
        } else if (fieldId === ORIGINAL_TIME_ESTIMATE) {
            return ORIGINAL_ESTIMATE_FORM_PARAM_ID;
        }
    }

    return TimeTrackingFieldsHelper;
});