AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');

    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    // data for ScrumBase and KanPlanBase decide if it can handle the data or not
    GH.RapidViewConfig.currentData.data = {
        sprintSupportEnabled: true
    };

    module("BacklogStatistics");

    test("module 'jira-agile/rapid/ui/plan/backlog-statistics' exists", function () {
        ok(BacklogStatistics);
        ok(GH.BacklogStatistics);
    });

    test('Test getAllIssues and getAllIssuesBeforeRankable', function () {
        var issues = GH.IssueTest.issueObjectsFromKeys(['GHS-1', 'GHS-2', 'GHS-3']);
        issues[0].hidden = true;
        var issueListModel = new IssueListModel('test', issues);
        issueListModel.insertMarker(5, false);

        var allIssues = issueListModel.getAllIssues();
        equal(_.keys(allIssues).length, 3);
        equal(allIssues['GHS-1'], issues[0]);
        equal(allIssues['GHS-2'], issues[1]);
        equal(allIssues['GHS-3'], issues[2]);

        var order = issueListModel.getOrder();
        var allIssuesBefore = BacklogStatistics.getAllIssuesBeforeIndex(allIssues, order, _.indexOf(order, 5));
        deepEqual(allIssuesBefore, []);

        allIssuesBefore = BacklogStatistics.getAllIssuesBeforeIndex(allIssues, order, _.indexOf(order, 'GHS-1'));
        deepEqual(allIssuesBefore, []);

        allIssuesBefore = BacklogStatistics.getAllIssuesBeforeIndex(allIssues, order, _.indexOf(order, 'GHS-2'));
        deepEqual(allIssuesBefore, [issues[0]]);

        allIssuesBefore = BacklogStatistics.getAllIssuesBeforeIndex(allIssues, order, _.indexOf(order, 'GHS-3'));
        deepEqual(allIssuesBefore, [issues[0], issues[1]]);

        issueListModel.insertMarker(5, 'GHS-3');
        order = issueListModel.getOrder();

        allIssuesBefore = BacklogStatistics.getAllIssuesBeforeIndex(allIssues, order, _.indexOf(order, 5));
        deepEqual(allIssuesBefore, issues);
    });

    test("correctly calculates issue stats when all issues are hidden", function () {
        var issues = [GH.IssueTest.createIssueObject({ hidden: true }), GH.IssueTest.createIssueObject({ hidden: true })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));

        var issueStats = BacklogStatistics.getIssueListStats(issueList, false);

        equal(issueStats.totalCount, 2);
        equal(issueStats.visibleCount, 0);
        equal(issueStats.allVisible, false);
        equal(issueStats.noneVisible, true);
        equal(issueStats.hasIssues, true);
        equal(issueStats.empty, false);
    });

    test("correctly calculates issue stats when there are no issues", function () {
        var issueList = new IssueListModel("model", [], new GH.VersionsModel([]));

        var issueStats = BacklogStatistics.getIssueListStats(issueList, false);

        equal(issueStats.totalCount, 0);
        equal(issueStats.visibleCount, 0);
        equal(issueStats.allVisible, true);
        equal(issueStats.noneVisible, true);
        equal(issueStats.hasIssues, false);
        equal(issueStats.empty, true);
    });

    test("correctly calculates issue stats when all issues are visible", function () {
        var issues = [GH.IssueTest.createIssueObject(), GH.IssueTest.createIssueObject()];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));

        var issueStats = BacklogStatistics.getIssueListStats(issueList, false);

        equal(issueStats.totalCount, 2);
        equal(issueStats.visibleCount, 2);
        equal(issueStats.allVisible, true);
        equal(issueStats.noneVisible, false);
        equal(issueStats.hasIssues, true);
        equal(issueStats.empty, false);
    });

    test("correctly calculates issue stats when some issues are filtered", function () {
        var issues = [GH.IssueTest.createIssueObject(), GH.IssueTest.createIssueObject(), GH.IssueTest.createIssueObject({ hidden: true })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));

        var issueStats = BacklogStatistics.getIssueListStats(issueList, false);

        equal(issueStats.totalCount, 3);
        equal(issueStats.visibleCount, 2);
        equal(issueStats.allVisible, false);
        equal(issueStats.noneVisible, false);
        equal(issueStats.hasIssues, true);
        equal(issueStats.empty, false);
    });

    test('Calculates total estimate of 0 when there are no issues', function () {
        BacklogModel.estimationStatistic = {
            isEnabled: true,
            name: "Story Points"
        };

        var issueList = new IssueListModel("model", [], new GH.VersionsModel([]));
        var stats = BacklogStatistics.getIssueListEstimateStatistics(issueList);

        ok(stats.estimate.shown);
        equal(stats.estimate.name, "Story Points");
        equal(stats.estimate.total.raw, 0);
        equal(stats.estimate.visible.raw, 0);
        ok(!stats.tracking.shown);
    });

    test('Calculates total estimate when all issues are visible', function () {
        BacklogModel.estimationStatistic = {
            isEnabled: true,
            name: "Story Points"
        };

        var issues = [GH.IssueTest.createIssueObjectWithEstimate(3), GH.IssueTest.createIssueObjectWithEstimate(5)];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));
        var stats = BacklogStatistics.getIssueListEstimateStatistics(issueList);

        ok(stats.estimate.shown);
        equal(stats.estimate.name, "Story Points");
        equal(stats.estimate.total.raw, 8);
        equal(stats.estimate.visible.raw, 8);
        ok(!stats.tracking.shown);
    });

    test('Calculates total estimate when some issues are visible', function () {
        BacklogModel.estimationStatistic = {
            isEnabled: true,
            name: "Story Points"
        };

        var issues = [GH.IssueTest.createIssueObjectWithEstimate(3), GH.IssueTest.createIssueObjectWithEstimate(5), GH.IssueTest.createIssueObjectWithEstimate(2, { hidden: true })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));
        var stats = BacklogStatistics.getIssueListEstimateStatistics(issueList);

        ok(stats.estimate.shown);
        equal(stats.estimate.name, "Story Points");
        equal(stats.estimate.total.raw, 10);
        equal(stats.estimate.visible.raw, 8);
        ok(!stats.tracking.shown);
    });

    test('getAssignedWorkStats aggregates estimate values by assignee', function () {
        var issues = [GH.IssueTest.createIssueObjectWithEstimate(3, { assignee: 'a', assigneeName: 'Alice' }), GH.IssueTest.createIssueObjectWithEstimate(null, { assignee: 'a', assigneeName: 'Alice' }), GH.IssueTest.createIssueObjectWithEstimate(null, { assignee: 'a', assigneeName: 'Alice' }), GH.IssueTest.createIssueObjectWithEstimate(5, { assignee: 'b', assigneeName: 'Bob' }), GH.IssueTest.createIssueObjectWithEstimate(2, { assignee: 'b', assigneeName: 'Bob' }), GH.IssueTest.createIssueObjectWithEstimate(2, { assignee: '', assigneeName: '' })];
        var issueList = new IssueListModel('model', issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);
        strictEqual(assignedWorkStats.assigned.length, 2, 'Two assignees');
        strictEqual(assignedWorkStats.assigned[0].totalEstimate, 7, 'Largest total estimate first');
        strictEqual(assignedWorkStats.assigned[0].issues.length, 2, 'Issues assigned to the first assignee');
        strictEqual(assignedWorkStats.assigned[1].totalEstimate, 3);
        strictEqual(assignedWorkStats.assigned[1].issues.length, 3, 'Issues assigned to the second assignee');
        strictEqual(assignedWorkStats.unassigned.totalEstimate, 2, 'Total unassigned work');
        strictEqual(assignedWorkStats.unassigned.issues.length, 1, 'Unassigned issues');
    });

    test('getAssignedWorkStats sorts assignees alphabetically when total estimate is equal', function () {
        var issues = [GH.IssueTest.createIssueObjectWithEstimate(3, { assignee: 'a', assigneeName: 'Z' }), GH.IssueTest.createIssueObjectWithEstimate(1, { assignee: 'a', assigneeName: 'Z' }), GH.IssueTest.createIssueObjectWithEstimate(3, { assignee: 'b', assigneeName: 'Y' }), GH.IssueTest.createIssueObjectWithEstimate(1, { assignee: 'b', assigneeName: 'Y' })];
        var issueList = new IssueListModel('model', issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);
        strictEqual(assignedWorkStats.assigned.length, 2, 'Two assignees');
        strictEqual(assignedWorkStats.assigned[0].assignee, 'b', 'Assignees with the same total estimate are sorted alphabetically');
        strictEqual(assignedWorkStats.assigned[1].assignee, 'a', 'Assignees with the same total estimate are sorted alphabetically');
    });

    test('getAssignedWorkStats ignores hidden issues', function () {
        var issues = [GH.IssueTest.createIssueObjectWithEstimate(3, { assignee: 'a', assigneeName: 'Alice', hidden: true }), GH.IssueTest.createIssueObjectWithEstimate(5, { assignee: 'b', assigneeName: 'Bob' }), GH.IssueTest.createIssueObjectWithEstimate(2, { assignee: 'b', assigneeName: 'Bob', hidden: true }), GH.IssueTest.createIssueObjectWithEstimate(2, { assignee: '', assigneeName: '' }), GH.IssueTest.createIssueObjectWithEstimate(3, { assignee: '', assigneeName: '', hidden: true })];
        var issueList = new IssueListModel('model', issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);
        strictEqual(assignedWorkStats.assigned.length, 1, 'One assignee after excluding hidden issues');
        strictEqual(assignedWorkStats.assigned[0].totalEstimate, 5, 'Total estimate excludes hidden issues');
        strictEqual(assignedWorkStats.unassigned.totalEstimate, 2, 'Total estimate excludes hidden issues for "unassigned"');
        strictEqual(assignedWorkStats.all.totalEstimate, 7, 'Total estimate excludes hidden issues for "all"');
    });

    test('getAssignedWorkStats includes assignees even if none of their issues are estimated', function () {
        var issues = [GH.IssueTest.createIssueObjectWithEstimate(null, { assignee: 'a', assigneeName: 'Alice' }), GH.IssueTest.createIssueObject({ assignee: 'b', assigneeName: 'Bob' }), GH.IssueTest.createIssueObject({ assignee: '', assigneeName: '' })];
        var issueList = new IssueListModel('model', issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);
        strictEqual(assignedWorkStats.assigned.length, 2, 'Two assignees');
        strictEqual(assignedWorkStats.assigned[0].totalEstimate, 0, 'Total estimate is 0');
        strictEqual(assignedWorkStats.assigned[1].totalEstimate, 0, 'Total estimate is 0');
        strictEqual(assignedWorkStats.unassigned.totalEstimate, 0, 'Total estimate is 0');
        strictEqual(assignedWorkStats.all.totalEstimate, 0, 'Total estimate is 0');
    });

    test('getAssignedWorkStats has totalTrackingEstimate when trackingStatistic is enabled', function () {
        BacklogModel.estimationStatistic = {
            isEnabled: true,
            name: "Original Time Estimate",
            renderer: "duration"
        };
        BacklogModel.trackingStatistic.isEnabled = true;

        var issues = [GH.IssueTest.createIssueObjectWithEstimate(14400, { assignee: 'a',
            assigneeName: 'Alice',
            trackingStatistic: {
                statFieldValue: {
                    value: 7200
                }
            } }), GH.IssueTest.createIssueObjectWithEstimate(14400, { assignee: 'a',
            assigneeName: 'Alice',
            trackingStatistic: {
                statFieldValue: {
                    value: 7200
                }
            } }), GH.IssueTest.createIssueObjectWithEstimate(14400, { assignee: 'b', assigneeName: 'Bob' })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);

        strictEqual(assignedWorkStats.assigned.length, 2, 'Two assignees');
        strictEqual(assignedWorkStats.assigned[0].totalEstimateText, '1d', 'Total estimate is 1d');
        strictEqual(assignedWorkStats.assigned[0].totalTrackingEstimateText, '14,400', 'Total tracking estimate is 4h');
        strictEqual(assignedWorkStats.assigned[1].totalEstimateText, '4h', 'Total estimate is 4h');
    });

    test('getAssignedWorkStats does not have totalTrackingEstimate when trackingStatistic is disabled', function () {
        BacklogModel.estimationStatistic = {
            isEnabled: true,
            name: "Original Time Estimate",
            renderer: "duration"
        };
        BacklogModel.trackingStatistic.isEnabled = false;

        var issues = [GH.IssueTest.createIssueObjectWithEstimate(14400, { assignee: 'a',
            assigneeName: 'Alice'
        }), GH.IssueTest.createIssueObjectWithEstimate(14400, { assignee: 'a',
            assigneeName: 'Alice'
        })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));
        var assignedWorkStats = BacklogStatistics.getAssignedWorkStats(issueList);

        strictEqual(assignedWorkStats.assigned.length, 1, 'One assignee');
        strictEqual(assignedWorkStats.assigned[0].totalEstimateText, '1d', 'Total estimate is 1d');
        ok(!('totalTrackingEstimateText' in assignedWorkStats.assigned[0]));
    });

    test("calculateSprintProgress shouldn't calculate hidden issues", function () {
        var issues = [GH.IssueTest.createIssueObjectWithEstimate(1, { hidden: false }), GH.IssueTest.createIssueObjectWithEstimate(2, { hidden: false }), GH.IssueTest.createIssueObjectWithEstimate(2, { hidden: true })];
        var issueList = new IssueListModel("model", issues, new GH.VersionsModel([]));
        var columns = [{ statusIds: [1] }];
        var estimationStatisticField = { isEnabled: true, typeId: "field", name: "Story Points", renderer: "number" };
        var issueStats = BacklogStatistics.calculateSprintProgress(issueList, columns, estimationStatisticField);

        equal(issueStats[0].total.value, 3);
    });

    module('calculateStatisticsFieldValue', {
        setup: function setup() {
            var issues = [GH.IssueTest.createIssueObject({ id: 1, 'hidden': false }), GH.IssueTest.createIssueObject({ id: 2, 'hidden': false, parentId: 1 }), GH.IssueTest.createIssueObject({ id: 3, 'hidden': true })];
            this.issueListModel = new IssueListModel("model", issues, new GH.VersionsModel([]));
        },
        teardown: function teardown() {
            this.issueListModel = null;
        }
    });

    test('calculateStatisticsFieldValue() should filter out subtasks when statisticsConfig is excluding subtasks', function () {
        var statisticsConfig = {
            typeId: BacklogStatistics.EXCLUDE_SUBTASKS_COLUMN_CONSTRAINT
        };
        var statisticsFieldValue = BacklogStatistics.calculateStatisticsFieldValue(this.issueListModel, statisticsConfig);
        deepEqual(statisticsFieldValue, {
            "statisticsFieldTotal": 2,
            "statisticsFieldVisible": 1,
            "excludeSubtasks": true,
            "subtasksTotal": 1,
            "subtasksVisible": 1
        });
    });

    test('calculateStatisticsFieldValue() should not filter out subtasks when statisticsConfig is not excluding subtasks', function () {
        var statisticsConfig = {
            typeId: 'none'
        };
        var statisticsFieldValue = BacklogStatistics.calculateStatisticsFieldValue(this.issueListModel, statisticsConfig);
        deepEqual(statisticsFieldValue, { "statisticsFieldTotal": 3, "statisticsFieldVisible": 2 });
    });
});