/* globals
 * GH.ContextMenu, GH.WorkContextMenuController, GH.RapidBoard.State
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-context-menu-controller
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/issue-actions
 * @requires module:jira-agile/rapid/ui/print/card-printing-controller
 */
define('jira-agile/rapid/ui/plan/plan-context-menu-controller', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var IssueActions = require('jira-agile/rapid/ui/plan/issue-actions');
    var CardPrintingController = require('jira-agile/rapid/ui/print/card-printing-controller');
    var SplitStoryAction = require('jira-agile/rapid/ui/plan/contextmenu/split-issue-action');
    var BacklogController;
    var BacklogSelectionController;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    });

    var PlanContextMenuController = {};
    PlanContextMenuController.ctxMenu = null;
    PlanContextMenuController.Analytics = {};
    PlanContextMenuController.Analytics.show = new AnalyticsTracker("gh.issue.ctx.menu.show");
    PlanContextMenuController.Analytics.action = new AnalyticsTracker("gh.issue.ctx.menu.action");
    /**
     * Resolves the current issue context based on the selected issues in the BacklogSelectionController.
     * There are 2 types of context: 'active-sprint' or 'backlog'
     */
    PlanContextMenuController.resolveContext = function () {
        var selectedIssueKeys = BacklogSelectionController.getSelectedAndVisibleIssuesInOrder();

        // find the model for the selected issue
        var currentModel = BacklogModel.findModelWithIssue(_.first(selectedIssueKeys));
        if (!currentModel) {
            return;
        }

        // fetch the data models
        var sprintModels = BacklogModel.getSprintModels();
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var hasBulkChangePermission = BacklogModel.hasBulkChangePermission;

        // get the data for the selected issues
        var issues = _.chain(currentModel.getIssueList().getAllIssues()).pick(selectedIssueKeys).values().value();

        var containsSubtasks = _.any(selectedIssueKeys, function (key) {
            var issue = currentModel.getIssueList().getIssueData(key);
            if (issue.parentId) {
                return true;
            }
        });

        var ctx = {
            containsSubtasks: containsSubtasks,
            singleIssue: selectedIssueKeys.length === 1,
            selectedIssues: selectedIssueKeys,
            selectedIssuesData: issues,
            sprintModels: sprintModels,
            backlogModel2: backlogModel2,
            currentModel: currentModel,
            flaggedIssuesCount: _.where(issues, { flagged: true }).length,
            hasBulkChangePermission: hasBulkChangePermission
        };

        return ctx;
    };

    /**
     * Initializes a context menu for use as an issue context menu and shows it at the given event's coordinates
     * @param event
     */
    PlanContextMenuController.showContextMenuAt = function (event) {
        PlanContextMenuController.hideContextMenu();

        // don't open the context menu when the user clicks on a link.
        if ($(event.target).closest('a').length > 0) {
            return;
        }

        var ctx = PlanContextMenuController.resolveContext();
        var menuOptions = {
            actionGroups: PlanContextMenuController.resolveActionsForContext(ctx)
        };
        PlanContextMenuController.ctxMenu = new GH.ContextMenu(menuOptions);
        PlanContextMenuController.ctxMenu.showAt(event);

        // prevent native context menu from showing
        event.preventDefault();
        // bring focus to the window, the preventDefault action might not have brought focus to the window
        $(GH.ContextMenu.menuSelector).focus();

        PlanContextMenuController.Analytics.show.trigger(ctx.type, "selectedIssuesCount", ctx.selectedIssues.length);
    };

    /**
     * Returns an array of action group objects relevant to the given context
     * @param ctx
     */
    PlanContextMenuController.resolveActionsForContext = function (ctx) {
        var actionGroups = [];
        var sendToActions = [];

        if (ctx.selectedIssues.length === 0) {
            return actionGroups;
        }

        // for each open sprint, add an action
        _.each(ctx.sprintModels, function (sprintModel) {
            // skip current model
            if (sprintModel.getId() === ctx.currentModel.getId()) {
                return;
            }

            var name = sprintModel.isActiveSprint() ? "send to active sprint" : "send to future sprint";

            sendToActions.push({
                id: "ghx-issue-ctx-action-send-to-sprint-" + sprintModel.getSprintId(),
                name: name,
                label: sprintModel.getSprintData().name,
                actionFn: function actionFn() {
                    var issueAction = ctx.containsSubtasks ? IssueActions.sendIssuesToSprint : IssueActions.sendIssuesToBottomOfSprint;
                    issueAction(ctx.selectedIssues, sprintModel).done(function () {
                        PlanContextMenuController.Analytics.action.trigger(name, "selectedIssuesCount", ctx.selectedIssues.length);
                    });
                }
            });
        });

        // Send to Backlog (without ranking)
        if (ctx.containsSubtasks) {
            if (ctx.backlogModel2.getId() !== ctx.currentModel.getId()) {
                sendToActions.push({
                    id: "ghx-issue-ctx-action-send-backlog",
                    name: "send to backlog",
                    label: AJS.I18n.getText("gh.rapid.plan.backlog.name"),
                    actionFn: function actionFn() {
                        IssueActions.sendIssuesToBacklog(ctx.selectedIssues);
                        PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                    }
                });
            }
        } else {
            sendToActions.push({
                id: "ghx-issue-ctx-action-send-top-backlog",
                name: "send to top of backlog",
                label: AJS.I18n.getText("gh.rapid.operations.send.top.backlog"),
                /*
                 shortcut: {
                 keys: ["s", "t"],
                 separator: '-'
                 },
                 */
                actionFn: function actionFn() {
                    IssueActions.sendIssuesToTopOfBacklog(ctx.selectedIssues, ctx.backlogModel2);
                    PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                }
            });

            sendToActions.push({
                id: "ghx-issue-ctx-action-send-bottom-backlog",
                name: "send to bottom of backlog",
                label: AJS.I18n.getText("gh.rapid.operations.send.bottom.backlog"),
                /*
                 shortcut: {
                 keys: ["s", "b"],
                 separator: '-'
                 },
                 */
                actionFn: function actionFn() {
                    IssueActions.sendIssuesToBottomOfBacklog(ctx.selectedIssues, ctx.backlogModel2);
                    PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                }
            });
        }

        if (BacklogModel.isRankable() && sendToActions.length) {
            actionGroups.push({
                label: AJS.I18n.getText("gh.rapid.operations.send.to"),
                actions: sendToActions
            });
        }

        var viewInIssueNavigatorAction = {
            id: "ghx-issue-ctx-action-view-in-navigator",
            name: "view in navigator",
            label: AJS.I18n.getText("gh.rapid.operations.view.in.navigator"),
            actionFn: function actionFn() {
                IssueActions.viewInIssueNavigator(ctx.selectedIssues);
                PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
            }
        };

        var viewInExcelAction = {
            id: "ghx-issue-ctx-action-view-in-excel",
            name: "view in excel",
            label: AJS.I18n.getText("gh.rapid.operations.view.in.excel"),
            actionFn: function actionFn() {
                IssueActions.viewInExcel(ctx.selectedIssues);
                PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
            }
        };

        var bulkChangeAction = {
            id: "ghx-issue-ctx-action-bulk-change",
            name: "bulk change",
            label: AJS.I18n.getText("gh.rapid.operations.bulk.change"),
            actionFn: function actionFn() {
                IssueActions.bulkChange(ctx.selectedIssues);
                PlanContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
            }

        };

        if (ctx.singleIssue) {
            var deleteAction = {
                id: "ghx-issue-ctx-action-delete",
                name: "delete",
                label: AJS.I18n.getText("gh.rapid.operations.delete"),
                actionFn: function actionFn() {
                    // spawn delete dialog
                    GH.IssueOperationShortcuts.deleteSelectedIssue();

                    // trigger analytics
                    AJS.trigger("analytics", { name: "gh.issue.ctx.menu.action.delete" });
                }
            };
            actionGroups.push({
                actions: [deleteAction]
            });
        }

        var flagAction;
        var flagAndCommentAction;
        if (ctx.flaggedIssuesCount !== ctx.selectedIssues.length) {
            // propose flag
            flagAction = {
                id: "ghx-issue-ctx-action-flag-toggle",
                name: "flag",
                label: AJS.I18n.getText("gh.rapid.operations.flag"),
                actionFn: function actionFn() {
                    IssueActions.flag(ctx.selectedIssues, true).done(function () {
                        BacklogController.loadData();
                    });
                    AJS.trigger("analytics", { name: "gh.issue.ctx.menu.action.flag", data: { selectedIssueCount: ctx.selectedIssues.length } });
                }
            };

            flagAndCommentAction = {
                id: "ghx-issue-ctx-action-flag-and-comment",
                name: "flag-and-comment",
                label: AJS.I18n.getText("gh.rapid.operations.flag.and.comment"),
                actionFn: function actionFn() {
                    GH.IssueFlagAndCommentAction.execute(ctx.selectedIssues, true, GH.BacklogController.loadData);
                    AJS.trigger("analytics", { name: "gh.issue.ctx.menu.action.flag.with.comment", data: { selectedIssueCount: ctx.selectedIssues.length } });
                }
            };
        } else {
            // propose unflag only if all the selected issues are already flagged
            flagAction = {
                id: "ghx-issue-ctx-action-unflag",
                name: "unflag",
                label: AJS.I18n.getText("gh.rapid.operations.unflag"),
                actionFn: function actionFn() {
                    IssueActions.flag(ctx.selectedIssues, false).done(function () {
                        BacklogController.loadData();
                    });
                    AJS.trigger("analytics", { name: "gh.issue.ctx.menu.action.unflag", data: { selectedIssueCount: ctx.selectedIssues.length } });
                }
            };

            flagAndCommentAction = {
                id: "ghx-issue-ctx-action-unflag-and-comment",
                name: "unflag-and-comment",
                label: AJS.I18n.getText("gh.rapid.operations.unflag.and.comment"),
                actionFn: function actionFn() {
                    GH.IssueFlagAndCommentAction.execute(ctx.selectedIssues, false, GH.BacklogController.loadData);
                    AJS.trigger("analytics", { name: "gh.issue.ctx.menu.action.unflag.with.comment", data: { selectedIssueCount: ctx.selectedIssues.length } });
                }
            };
        }

        actionGroups.push({
            actions: [flagAction, flagAndCommentAction]
        });

        SplitStoryAction.appendMenuItem(ctx, actionGroups);

        actionGroups.push({
            actions: [viewInIssueNavigatorAction, viewInExcelAction]
        });

        if (ctx.hasBulkChangePermission) {
            actionGroups.push({
                actions: [bulkChangeAction]
            });
        }

        if (ctx.singleIssue) {
            var printCardsAction = {
                id: "ghx-issue-ctx-action-print-cards",
                name: "print cards",
                label: AJS.I18n.getText("gh.rapid.operations.print.selected.cards", ctx.selectedIssues.length),
                actionFn: function actionFn() {
                    CardPrintingController.printSelectedIssues(prepareDataToPrintSelectedIssues(ctx.selectedIssues));
                }
            };
            actionGroups.push({
                actions: [printCardsAction]
            });
        }

        return actionGroups;
    };

    function prepareDataToPrintSelectedIssues(issueKeys) {
        var sprintModels = BacklogModel.getSprintModels();
        var issuesToPrint = [];

        var findInSprints = function findInSprints(key) {
            var issue = null;
            _.find(sprintModels, function (sprintModel) {
                issue = sprintModel.getIssueList().getAllIssues()[key];
                return !!issue;
            });
            return issue || null;
        };

        var findInBacklog = function findInBacklog(key) {
            var backlogIssueList = BacklogModel.getBacklogModel2().getIssueList();
            return backlogIssueList.getAllIssues()[key];
        };

        // order of issues to be printed obeys the order of issue keys provided
        _.each(issueKeys, function (key) {
            // search for this issue in list of sprints, stop at the first found sprint
            var issue = findInSprints(key) || findInBacklog(key);
            if (issue) {
                issuesToPrint.push(issue);
            }
        });

        return {
            issues: issuesToPrint,
            epics: BacklogController.getEpicMap(),
            viewMode: 'plan',
            boardType: GH.RapidBoard.State.getBoardType()
        };
    }

    PlanContextMenuController.hideContextMenu = function () {
        if (PlanContextMenuController.ctxMenu != null) {
            PlanContextMenuController.ctxMenu.hide();
            PlanContextMenuController.ctxMenu = null;
        }
    };

    return PlanContextMenuController;
});