/* globals
 * GH, GH.RapidBoard, GH.EpicController, GH.Notification, GH.VersionController,
 * GH.ConfirmDialog, GH.tpl, GH.EpicView, GH.VersionView, GH.LinkedPagesController, GH.RankController, GH.DetailsView,
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-drag-and-drop
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/BacklogView
 * @requires module:jira-agile/rapid/ui/plan/issue-actions
 * @requires module:jira-agile/rapid/ui/plan/issue-move-controller
 * @requires module:jira-agile/rapid/ui/plan/kanban-transition-and-rank
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-controls
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-filtering
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 */

define('jira-agile/rapid/ui/plan/plan-drag-and-drop', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var DarkFeature = require('jira/ajs/dark-features');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var IssueActions = require('jira-agile/rapid/ui/plan/issue-actions');
    var IssueMoveController = require('jira-agile/rapid/ui/plan/issue-move-controller');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var KanbanTransitionAndRank = require('jira-agile/rapid/ui/plan/kanban-transition-and-rank');
    var PlanIssueListFiltering = require('jira-agile/rapid/ui/plan/plan-issue-list-filtering');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    var BacklogController;
    var BacklogSelectionController;
    var BacklogView;
    var PlanIssueListView;
    var SORTABLE_Z_INDEX = 3000;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
        BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
        PlanIssueListView = require('jira-agile/rapid/ui/plan/plan-issue-list-view');
    });

    var PlanDragAndDrop = {};

    /** Element that's being dragged */
    PlanDragAndDrop.currentDragElement = undefined;

    /**
     * Used by WebDriver tests to check whether the backlog is redrawn after ranking
     */
    PlanDragAndDrop.rankingComplete = true;

    /**
     * The data attribute name that is used droppable to set the reason of rejection.
     *
     * It is specifically used when dropping to get the reason why the rejection pre calculation (on start drag event)
     * excluded the set of dragged issues.
     *
     * @type {string}
     */
    PlanDragAndDrop.REJECTION_REASON_ATTRIBUTE = 'data-target-rejected-reason';

    /**
     * Constants to identify the reason of a drop rejection on versions.
     *
     * @type {{VERSION_PROJECT_MISMATCH: string, ISSUE_PROJECT_MISMATCH: string}}
     */
    PlanDragAndDrop.DropRejectionReasons = {
        VERSION_PROJECT_MISMATCH: 'versionProjectMismatch',
        ISSUE_PROJECT_MISMATCH: 'issueProjectMismatch'
    };

    PlanDragAndDrop.analytics = {};

    PlanDragAndDrop.analytics.sprints = new AnalyticsTracker("gh.sprint.issues");

    PlanDragAndDrop.analytics.ranking = new AnalyticsTracker("gh.rapidboard.rankissues");

    PlanDragAndDrop.cancel = false;
    PlanDragAndDrop.skipNextRank = false;

    /**
     * Get the selector for all elements on which a draggable action shouldn't start
     */
    PlanDragAndDrop.getCancelSelector = function () {
        var cancel = ':input,button,.js-fake-parent,.ghx-subtask-group,.ghx-parent-stub,.ghx-hidden-subtasks-count-container';

        var rankable = BacklogModel.isRankable();
        var canManageSprints = BacklogModel.canManageSprints();

        // register a mouse down handler for all issues to alert user about the non-rankability
        // when they try to rank an issue or move the marker.
        if (!rankable) {
            PlanDragAndDrop.NotRankableHandler.register('.js-sortable.js-issue');

            // and make sure they cant drag these things
            cancel += ',.js-issue';
        } else if (!canManageSprints) {
            cancel += ',.ghx-marker';
        }
        return cancel;
    };

    /**
     * This is a set of "custom" options commonly used by the sortable instances in backlog
     * so that it plays well with our UI layout (containers with scroll, sticky header, different stacking contexts etc),
     */

    PlanDragAndDrop.COMMON_CUSTOM_SORTABLE_OPTIONS = Object.freeze({
        zIndex: SORTABLE_Z_INDEX,
        appendTo: 'body',
        helper: 'clone',
        scroll: true,
        ignoreOriginalScroll: true,
        scrollSensitivity: 20,
        additionalScrollParents: function additionalScrollParents() {
            return $('#ghx-backlog, .ghx-classification-scrollview');
        },
        // This function is used to find the dynamic top auto scroll offset (for sticky header feature
        // since the height of headers vary)
        autoScrollOffsets: function autoScrollOffsets() {
            var $backlogColumn = $('#ghx-content-group');
            if (!BacklogView.isStickySprintHeadersEnabled()) {
                // If sticky headers are disabled, just return an empty object so that the default
                // scroll sensitivity will be used
                return {};
            }

            var $header = $backlogColumn.find('.ghx-backlog-header.stuck');
            var top = $header.length ? $header.outerHeight(true) : 0;
            return {
                top: top + 20
            };
        }
    });

    /**
     * This method returns the boundary in which subtasks can be dragged in backlog
     */
    PlanDragAndDrop.subtaskDnDContainment = function () {
        var DEFAULT_CONTAINMENT = 10000;
        var $backlogCointainer = $('.ghx-backlog-container.ghx-kanban-backlog');
        var $epicsContainer = $('#ghx-epic-column');
        var backlogBottom = $backlogCointainer.offset().top + $backlogCointainer.outerHeight();
        // It is assumed here that the versions column has the same height as epics column, so we only consider one of them
        var epicsBottom = $epicsContainer.offset().top + $epicsContainer.outerHeight();
        var rightBoundary = this.helper.width() * -1;

        return [isNaN(rightBoundary) ? 0 : rightBoundary, 0, DEFAULT_CONTAINMENT, Math.max(backlogBottom, epicsBottom) || DEFAULT_CONTAINMENT];
    };

    /**
     *
     */
    PlanDragAndDrop.enableDragAndDrop = function () {

        // fetch the backlog column
        var $backlogColumn = $('#ghx-content-group');

        var cancel = PlanDragAndDrop.getCancelSelector();

        PlanDragAndDrop.skipNextRank = false;

        var mousedownSearchBlur = function mousedownSearchBlur() {
            // GHS-5110 This is to make sure the blur handler is called on the search input box. See this SO:
            // http://stackoverflow.com/questions/8869708/click-on-jquery-sortable-list-does-not-blur-input
            // I just spent the better part of a day on this. Don't remove it unless you test GHS-5110 again.

            // Checking before calling blur to avoid repeat blur calls
            if (document.activeElement && document.activeElement.id === "ghx-backlog-search-input") {
                $('#ghx-backlog-search-input').blur();
            }
        };
        $backlogColumn.mousedown(function (evt) {
            // ignore if we are clicking on an inline edit!
            if ($(evt.target).is('.js-editing-field')) {
                return;
            }

            mousedownSearchBlur();
            if (GH.RapidBoard.Util.InlineEditable.submitActiveEdits()) {
                // prevents drag operation to start if there were some active inline edits
                evt.stopPropagation();
            }
        });

        // decide which elements should be draggable
        var $firstIssue = $backlogColumn.find('.js-issue').first();
        $backlogColumn.sortable(_.extend({}, PlanDragAndDrop.COMMON_CUSTOM_SORTABLE_OPTIONS, {
            revert: 0,
            forceHelperSize: true,
            helperProportions: { width: $firstIssue.width(), height: $firstIssue.height() },
            initializeTopAndLeft: true, // required because markers are y axis only and don't get a left value as a result...
            containment: function containment() {
                // no containment for issues
                if (!this.helper.hasClass('ghx-marker')) {
                    return false;
                }

                //var ce = this.helper.parent();
                var ce = this.currentItem.parent();
                var co = ce.offset();
                var il = ce.find('.js-issue-list');
                var io = il.offset();
                var header = ce.parent().find('.js-sprint-header');
                //            var meta = ce.find('.ghx-sprint-info');
                //var helper = ce.find('.ghx-helper');
                var topOffset = io.top;
                // If the header is stick to the top don't drag over it
                if (header.hasClass('stuck')) {
                    topOffset = header.offset().top + header.outerHeight();
                }
                return [0, topOffset, 10000, 10000];
            },
            items: '.js-parent-drag:not(.ghx-filtered)',
            // don't drag on the fake parent and the subtask group (activates in IE8 only)
            cancel: cancel,
            intersectionOverride: function intersectionOverride(item) {
                var isOverElementHeight = $.ui.isOverAxis(this.positionAbs.top + this.helperProportions.height / 2, item.top, item.height),
                    isOverElementWidth = $.ui.isOverAxis(this.positionAbs.left + this.offset.click.left, item.left, item.width),
                    isOverElement = isOverElementHeight && isOverElementWidth,
                    verticalDirection = this._getDragVerticalDirection(),
                    horizontalDirection = this._getDragHorizontalDirection();

                if (!isOverElement) {
                    return false;
                }

                return this.floating ? horizontalDirection && horizontalDirection == "right" || verticalDirection == "down" ? 2 : 1 : verticalDirection && (verticalDirection == "down" ? 2 : 1);
            },
            helperAdjusted: function helperAdjusted() {
                if (PlanDragAndDrop.markerMove) {
                    PlanDragAndDrop.adjustMarkerDragOverlayHeight.apply(this);
                }
            },
            axisAdjusted: function axisAdjusted() {
                // markers are y axis restricted, issues y axis restricted if epics support isn't enabled
                var axis = 'y';
                if (this.currentItem.is('.js-issue')) {
                    axis = 'both';
                }
                this.options.axis = axis;
            },
            start: function start(event, ui) {
                PlanDragAndDrop.skipNextRank = false;
                $backlogColumn.addClass('ghx-drag-in-progress');

                PlanDragAndDrop.dragStartHandler.apply(this, [event, ui]);
            },
            // if we've just had a droppable drop, then skip this rank
            update: function update(event, ui) {
                if (PlanDragAndDrop.skipNextRank) {
                    $backlogColumn.sortable('cancel');
                } else {
                    PlanDragAndDrop.updateHandler.apply(this, [event, ui]);
                }
            },
            stop: function stop(event, ui) {
                $backlogColumn.removeClass('ghx-drag-in-progress');

                PlanDragAndDrop.dragStopHandler.apply(this, [event, ui]);
            },
            change: PlanDragAndDrop.changeHandler,
            disabled: !BacklogModel.isRankable()
        }));

        /**
         * This is for accepting drops on sprint headers of closed sprints
         * as well as for accepting drop of subtasks in both sprint and backlog since subtasks aren't valid
         * sortable items of the main sortable instance (#ghx-content-group)
         *
         * Sub task drag and drop can be implemented using "connectWith" option of sortable but for some reason
         * connected sortable doesn't receive the item at the moment, hence droppable is used
         */

        function shouldUseContainerDropzone($container) {
            return $container.find('.ghx-dropzone-active').length || !$container.hasClass('ghx-open') || $container.find('.ghx-no-issues').length;
        }

        $('.ghx-backlog-container').droppable({
            accept: '.js-sortable.js-issue',
            hoverClass: 'ghx-target-hover',
            tolerance: 'pointer',
            activate: function activate(e, ui) {
                var $this = $(this);
                var isSubTask = ui.draggable.hasClass('ghx-subtask');
                // Display the drop zone in other sprints, only while dragging subtasks
                if (!isSubTask || $this.has(ui.draggable).length) {
                    return;
                }
                $this.find('.js-issue-list .ghx-plan-dropzone').addClass('ghx-dropzone-active');
            },
            deactivate: function deactivate(e, ui) {
                $(this).find('.js-issue-list .ghx-plan-dropzone').removeClass('ghx-dropzone-active');
            },
            drop: function drop(event, ui) {
                var $this = $(this);
                var isBacklog = $this.hasClass('ghx-everything-else');

                if (!shouldUseContainerDropzone($this) || PlanDragAndDrop.cancel) {
                    return;
                }

                var selectedIssueKeys = BacklogSelectionController.getSelectedIssueKeys();
                selectedIssueKeys = BacklogSelectionController.filterIssuesThatCanBeRanked(selectedIssueKeys);
                var sprintId = isBacklog ? null : parseInt($(this).attr('data-sprint-id'), 10);
                var sprintModel = isBacklog ? BacklogModel.getBacklogModel2() : BacklogModel.getSprintModel(sprintId);
                if (!sprintModel) {
                    return;
                }

                // fire analytics
                PlanDragAndDrop.analytics.sprints.trigger("addbydragndrop", "selectedIssuesCount", selectedIssueKeys.length); // SAFE

                // add the issues to the sprint
                var lastIssueKey = _.last(sprintModel.getIssueList().getOrder());
                PlanDragAndDrop.rankIssues(selectedIssueKeys, sprintId, lastIssueKey, null);

                // This looks racy, but it's the least nasty way of ensuring that the rank operation doesn't happen.
                // jquery ui is not great for this... We should be able to call backlogColumn.sortable('cancel');
                // here instead, but that makes jQuery blow up.
                PlanDragAndDrop.skipNextRank = true;
            }
        });

        //sub-tasks DnD
        $backlogColumn.find('.ghx-subtask-group').sortable(_.extend({}, PlanDragAndDrop.COMMON_CUSTOM_SORTABLE_OPTIONS, {
            revert: 0,
            items: '.ghx-subtask',
            /*
             * containment in the form of [x1, y1, x2, y2]. This is added to prevent users from scrolling down indefinitely
             * which causes issues with placeholder position, which in turn affects the DnD
            */
            containment: PlanDragAndDrop.subtaskDnDContainment,
            start: function start(event, ui) {
                PlanDragAndDrop.skipNextRank = false;
                $backlogColumn.addClass('ghx-drag-in-progress');

                PlanDragAndDrop.dragStartHandler.apply(this, [event, ui]);
            },
            stop: function stop(event, ui) {
                $backlogColumn.removeClass('ghx-drag-in-progress');

                PlanDragAndDrop.dragStopHandler.apply(this, [event, ui]);
            },
            update: function update(event, ui) {
                if (PlanDragAndDrop.skipNextRank) {
                    $backlogColumn.sortable('cancel');
                } else {
                    PlanDragAndDrop.updateHandler.apply(this, [event, ui]);
                }
            }
        }));

        PlanDragAndDrop.initializeClassificationDnD();
    };

    PlanDragAndDrop.initializeClassificationDnD = function () {

        var cancel = PlanDragAndDrop.getCancelSelector();

        var $allClassificationCards = $('.ghx-classification-cards');
        var $epicsCards = $('#ghx-epic-column').find('.ghx-classification-cards');

        $epicsCards.sortable({
            // use clone to fix a FF 15 bug / behaviour that generates a click event on the sorted element
            // in epic case it would activate the filter
            helper: 'clone',
            axis: 'y',
            revert: 0,
            zIndex: SORTABLE_Z_INDEX,
            containment: 'parent',
            start: function start(event, ui) {
                PlanDragAndDrop.dragStartHandler.apply(this, [event, ui]);
            },
            stop: PlanDragAndDrop.dragStopHandler,
            tolerance: 'pointer',
            items: '.js-parent-drag, .js-sortable',
            // don't drag on the fake parent and the subtask group (activates in IE8 only)
            cancel: cancel,
            // if we've just had a droppable drop, then skip this rank
            update: function update(event, ui) {
                PlanDragAndDrop.updateHandler.apply(this, [event, ui]);
            }
            // Not needed for epicschange: PlanDragAndDrop.changeHandler
        });

        var $droppableElement = $allClassificationCards.find('.ghx-classification-item').add($('.ghx-classification-pending'));

        // Over/out droppable events are not always triggered in the right order
        // To prevent incorrect cursor style we use the balance of those events
        // Positive balance means we have to show a not allowed cursor icon
        var overOutEventsBalance = 0;

        $droppableElement.droppable({
            accept: '.js-sortable.js-issue',
            hoverClass: 'ghx-target-hover',
            tolerance: 'pointer',
            over: function over(event, ui) {
                if ($(this).hasClass('ghx-target-hover-rejected')) {
                    overOutEventsBalance++;
                    if (overOutEventsBalance > 0) {
                        // we use direct style affectation because IE is not able to apply css properties changes
                        // introduced by a class affectation after a when mouse button is pressed
                        ui.helper.css('cursor', 'not-allowed');
                    }
                }
            },
            out: function out(event, ui) {
                if ($(this).hasClass('ghx-target-hover-rejected')) {
                    overOutEventsBalance--;
                    if (overOutEventsBalance <= 0) {
                        // remove the style added in the over event handler
                        ui.helper.css('cursor', '');
                    }
                }
            },
            drop: function drop() {
                overOutEventsBalance = 0;

                var $classificationItem = $(this);
                var epicKey = $classificationItem.attr('data-epic-key');
                var versionId = $classificationItem.attr('data-version-id');

                // stop if the drag was canceled
                if (PlanDragAndDrop.cancel) {
                    return;
                }

                // fetch the selected keys and the epic key
                var selectedIssuesKeys = BacklogSelectionController.getSelectedIssueKeys();

                if (!_.isEmpty(epicKey)) {
                    if (PlanIssueListFiltering.isNoneEpicFilterKey(epicKey)) {
                        GH.EpicController.removeIssuesFromAssociatedEpics(selectedIssuesKeys);
                    } else {
                        GH.EpicController.addIssuesToEpic(epicKey, selectedIssuesKeys);
                    }
                } else if (!_.isEmpty(versionId)) {
                    if ($classificationItem.hasClass('ghx-target-hover-rejected')) {
                        // the target is actually not able to accept the draggable, this class was set during the start event
                        // of the jQuery UI draggable
                        var reason = $classificationItem.attr(PlanDragAndDrop.REJECTION_REASON_ATTRIBUTE);
                        if (reason === PlanDragAndDrop.DropRejectionReasons.VERSION_PROJECT_MISMATCH || reason === PlanDragAndDrop.DropRejectionReasons.ISSUE_PROJECT_MISMATCH) {
                            var message = AJS.I18n.getText('gh.version.issues.project.mismatch', selectedIssuesKeys.length);
                            GH.Notification.showError(null, message, true, { autoHide: true, showTitle: false });
                        }
                    } else {

                        var selectedIssues = _.compact(_.map(selectedIssuesKeys, function (issueKey) {
                            return BacklogModel.getIssueData(issueKey);
                        }));
                        var version = GH.VersionController.getVersionModel().getVersion(parseInt(versionId, 10));

                        // if we do not try to "clear" versions - filter out issues which belong to different project than the version itself
                        if (!PlanIssueListFiltering.isNoneVersionFilterId(versionId)) {
                            var filteredIssuesCount = selectedIssues.length;
                            selectedIssues = _.filter(selectedIssues, function (model) {
                                return version.project.id === model.projectId;
                            });
                            filteredIssuesCount -= selectedIssues.length;
                        }

                        // does any of the issues have multiple fix version ?
                        var multipleVersions = _.some(selectedIssues, function (issue) {
                            return issue.fixVersions.length > 1;
                        });

                        // the code that effectively update the issues
                        var action = function action() {
                            if (PlanIssueListFiltering.isNoneVersionFilterId(versionId)) {
                                GH.VersionController.clearIssuesVersion(selectedIssues);
                            } else {
                                GH.VersionController.setIssuesVersion(parseInt(versionId, 10), selectedIssues, filteredIssuesCount);
                            }
                        };

                        if (multipleVersions && !GH.storage.get('gh.dialog.versionChangeConfirm.dontAsk', false)) {
                            // any of the issues have multiple fix versions so we want a confirmation form the user

                            var confirmMessage;
                            if (PlanIssueListFiltering.isNoneVersionFilterId(versionId)) {
                                confirmMessage = AJS.I18n.getText('gh.version.issues.destructive.remove', selectedIssues.length);
                            } else {
                                confirmMessage = AJS.I18n.getText('gh.version.issues.destructive.assignment', '<b>' + AJS.escapeHTML(String(version.name)) + '</b>', selectedIssues.length);
                            }

                            var confirmCssId = 'ghx-version-change-confirm-dialog';

                            // create and show the confirm dialog
                            GH.ConfirmDialog.create(confirmCssId, {
                                content: confirmMessage,
                                contentEscaping: false,
                                onConfirmFn: function onConfirmFn() {
                                    action();
                                }
                            }).show();

                            // inject the checkbox in the footer since we currently have no way to do that with the API
                            $('#' + confirmCssId).find('.dialog-button-panel').prepend(GH.tpl.versionview.renderDontAskAgain());

                            var $dontAskCheckbox = $('#ghx-dont-ask-for-confirmation');
                            $dontAskCheckbox.change(function () {
                                GH.storage.put('gh.dialog.versionChangeConfirm.dontAsk', $dontAskCheckbox.is(':checked'), false);
                            });
                        } else {
                            // no confirmation needed
                            action();
                        }
                    }
                }

                // This looks racy, but it's the least nasty way of ensuring that the rank operation doesn't happen.
                // jquery ui is not great for this... We should be able to call backlogColumn.sortable('cancel');
                // here instead, but that makes jQuery blow up.
                PlanDragAndDrop.skipNextRank = true;
            }
        });
    };

    /**
     * Delegating drag start handler
     */
    PlanDragAndDrop.dragStartHandler = function (event, ui) {
        PlanDragAndDrop.cancel = false;
        var elem = $(ui.item);
        var container = $(this);
        var isDraggingMarker = elem.hasClass('ghx-marker') || elem.hasClass('js-parent-drag');

        // ensure we stop the search filter edits
        setTimeout(function () {
            PlanControls.stopEdits();
        }, 0);

        // Ensure we hide epic and version dropdowns
        GH.EpicView.hideDropdown();
        GH.VersionView.hideDropdown();

        GH.PlanContextMenuController.hideContextMenu();

        // bind a keyhandler for esc
        $(document).bind('keydown.planDragAndDrop', function (event) {
            if (event.keyCode == 27) {
                PlanDragAndDrop.cancel = true;
                container.sortable('cancel');

                // Redraw all statistics
                if (isDraggingMarker) {
                    BacklogController.revertStatistics();
                    GH.BacklogView.enableStickyHeaders();
                }
            }
        });

        if (elem.hasClass('js-issue')) {
            PlanDragAndDrop.markerMove = false;
            PlanDragAndDrop.issueDragStartHandler.apply(this, [event, ui]);
        } else if (isDraggingMarker) {
            PlanDragAndDrop.markerMove = true;
            PlanDragAndDrop.markerDragStartHandler.apply(this, [event, ui]);
        }

        GH.LinkedPagesController.hideCurrentDialog();
    };

    /**
     * Delegating drag stop handler
     */
    PlanDragAndDrop.dragStopHandler = function (event, ui) {
        var elem = $(ui.item);

        // unbind the key handler
        $(document).unbind('keydown.planDragAndDrop');

        if (elem.hasClass('js-issue')) {
            PlanDragAndDrop.issueDragStopHandler.apply(this, [event, ui]);
        } else if (elem.hasClass('ghx-marker') || elem.hasClass('js-parent-drag')) {
            PlanDragAndDrop.markerDragStopHandler.apply(this, [event, ui]);
        }
    };

    /**
     * Delegating update handler
     */
    PlanDragAndDrop.updateHandler = function (event, ui) {
        var elem = ui.item;

        // if we're in a cancel, fail fast (this still gets called, even when we 'cancel')
        if (PlanDragAndDrop.cancel) {
            return;
        }

        if (elem.hasClass('js-issue')) {
            PlanDragAndDrop.issueUpdateHandler(event, ui);
        } else if (elem.hasClass('ghx-marker') || elem.hasClass('js-parent-drag')) {
            PlanDragAndDrop.markerUpdateHandler(event, ui);
        } else if (elem.hasClass('ghx-classification-item')) {
            PlanDragAndDrop.epicUpdateHandler(event, ui);
        }
    };

    /**
     * Delegating change handler
     */
    PlanDragAndDrop.changeHandler = function (event, ui) {
        var elem = $(ui.item);
        var isIssue = elem.hasClass('js-issue');
        var $placeholder = ui.placeholder;
        var $placeholderParent = $placeholder.parent();
        var isInIssueList = $placeholderParent.hasClass('js-issue-list');

        // handle issue placement
        if (isIssue) {
            // issues can only be dragged into other js-issue-list containers
            if (!isInIssueList) {
                // move back into list
                var $prevIssueList = $placeholder.prevAll('.js-issue-list');
                if ($prevIssueList.length) {
                    $prevIssueList.append($placeholder);
                }
            }
        }

        // handle marker placement
        else {

                // find the sprint id of the dragged marker
                var draggedSprintId = SprintView.getSprintIdForMarker($(ui.item));
                var currentSprintId = SprintView.getSprintIdForElement($placeholder);
                var isInOriginSprint = draggedSprintId === currentSprintId;
                // different behaviour for current dragged sprint and other sprints
                if (isInOriginSprint) {

                    // original sprint: make sure marker is outside of list when placed at the end of the list
                    if (isInIssueList) {
                        // we have the empty description as last element - ignore that one!
                        var isLastElement = !$placeholder.next('.js-issue').length;
                        if (isLastElement) {
                            // move the placeholder to be after the list
                            $placeholderParent.after($placeholder);
                        }
                    }
                } else {
                    // other sprints: make sure the marker is always inside the issue list
                    if (!isInIssueList) {
                        // move back into list
                        var $prevIssueList = $placeholder.prevAll('.js-issue-list');
                        if ($prevIssueList.length) {
                            $prevIssueList.append($placeholder);
                        }
                    }
                }
            }

        // MARKER UPDATE CODE
        if (elem.hasClass('js-issue')) {
            // leave marker stats incorrect on issue move for now.
        } else if (elem.hasClass('ghx-marker') || elem.hasClass('js-parent-drag')) {
            PlanDragAndDrop.markerChangeHandler(event, ui);
        }
    };

    /**
     * Called when issue dragging starts.
     */
    PlanDragAndDrop.issueDragStartHandler = function (event, ui) {
        var helper = $(ui.helper);

        // Keep reference to draggable
        PlanDragAndDrop.currentDragElement = helper;

        // ensure that the issue of the card we are dragging is selected
        var issueKey = PlanIssueListView.getIssueKey(helper);
        BacklogSelectionController.ensureIssueSelected(event, issueKey);

        // fetch selected issues
        var selectedIssues = BacklogSelectionController.getSelectedIssueKeys();
        selectedIssues = BacklogSelectionController.filterIssuesThatCanBeRanked(selectedIssues);

        // handle multidrag
        if (selectedIssues.length > 1) {

            if (helper.hasClass("ghx-parent-group")) {
                helper.children('.js-issue').addClass('ghx-move-main').find('.ghx-move-count > b').text(selectedIssues.length);
            } else {
                helper.addClass('ghx-move-main').find('.ghx-move-count > b').text(selectedIssues.length);
            }
        }

        // check versions to detect which one we can drop onto
        var uniqueProjectIds = _.chain(selectedIssues).map(function (issueKey) {
            return BacklogModel.getIssueData(issueKey).projectId;
        }).uniq().value();

        var $versionsItems = $('#ghx-version-column').find('.ghx-classification-cards .ghx-classification-item');
        var versionsFromProjects = _.flatten(uniqueProjectIds.map(function (projectId) {
            return GH.VersionController.getVersionModel().getVersionsIdsForProject(projectId);
        }));

        // not droppable because the version and the issues don't belong to the same project
        $versionsItems.filter(function (index, versionItem) {
            return !_.contains(versionsFromProjects, parseInt($(versionItem).attr('data-version-id'), 10));
        }).addClass('ghx-target-hover-rejected').attr('data-target-rejected-reason', 'versionProjectMismatch');
    };

    /**
     * Called when an issue drag operation stopped.
     */
    PlanDragAndDrop.issueDragStopHandler = function (event, ui) {
        var card = ui.item;

        // remove the cursor style possibly added by the droppable over event handler
        card.css('cursor', '');

        // remove multi-drag label
        if (card.hasClass('ghx-parent-group')) {
            card.children('.js-issue').removeClass('ghx-move-main');
        } else {
            card.removeClass('ghx-move-main');
        }

        PlanDragAndDrop.currentDragElement = null;
        $('.ghx-classification-cards').find(".ghx-target-hover-rejected").removeClass('ghx-target-hover-rejected').removeAttr(PlanDragAndDrop.REJECTION_REASON_ATTRIBUTE);
    };

    /**
     * Handles the update event of sortables which are to be ranked.
     */
    PlanDragAndDrop.issueUpdateHandler = function (event, ui) {
        PlanDragAndDrop.rankingComplete = false;
        var $draggedIssue = $(ui.item);
        var issueKeys = BacklogSelectionController.getSelectedAndVisibleIssuesInOrder();

        // do analytics before we even know if they are successful
        PlanDragAndDrop.analytics.ranking.trigger('dragAndDrop', issueKeys.length); // SAFE

        // find the previous issue (might have to jump over a marker)
        var prevElement = PlanIssueListView.findPrevRankableElement($draggedIssue);
        var prevRankableId = PlanIssueListView.getRankableId(prevElement);

        // find the next issue (might have to jump over a marker)
        var nextElement = PlanIssueListView.findNextRankableElement($draggedIssue);
        var nextRankableId = PlanIssueListView.getRankableId(nextElement);

        // find the sprint the issue now is in
        var sprintId = SprintView.getSprintIdForElement($draggedIssue);

        // rather rank after prev than before next
        var hasPrev = !!prevRankableId;
        var hasNext = !!nextRankableId;
        if (hasPrev && hasNext) {
            nextRankableId = false;
        }

        PlanDragAndDrop.rankIssues(issueKeys, sprintId, prevRankableId, nextRankableId);
    };

    /**
     * Handles the update event of sortables which are to be ranked.
     */
    PlanDragAndDrop.epicUpdateHandler = function (event, ui) {
        function getKeyAndId(epic) {
            return { key: epic.attr('data-epic-key'), id: epic.attr('data-epic-id') };
        }

        var draggedEpic = $(ui.item);
        var epicKeyId = getKeyAndId(draggedEpic);

        // find the previous and next epic
        var prevEpic = draggedEpic.prev();
        var prevEpicKeyId = prevEpic.length > 0 ? getKeyAndId(prevEpic) : false;

        var nextEpic = draggedEpic.next();
        var nextEpicKeyId = nextEpic.length > 0 ? getKeyAndId(nextEpic) : false;

        // even before we know if ranking is successful, reorder issues in data model client side, then re-draw
        var issueKeys = [epicKeyId.key];
        GH.EpicController.reorderEpics(issueKeys, prevEpicKeyId.key, nextEpicKeyId.key);

        var errorFn = function errorFn() {
            // if ranking failed, reload model and redraw
            PlanController.reload();
        };

        var doneHandler = function doneHandler(response) {
            // inspect results for error response
            var errorRankResult = GH.RankController.getErrorRankResultFromResponse(response);
            if (errorRankResult) {
                GH.Notification.showWarning(errorRankResult.errors[0]);
                PlanController.reload();
            }
        };

        GH.RankController.rankEpics(BacklogModel.getRankCustomFieldId(), epicKeyId.id, nextEpicKeyId.id, prevEpicKeyId.id).fail(errorFn).done(doneHandler);
    };

    /**
     * Adds issues to a sprint and/or ranks them
     *
     * @param {Array.<string>} issueKeys
     * @param {number} sprintId
     * @param {number|null} prevRankableId
     * @param (number|null) nextRankableId
     * @returns {Deferred}
     */
    PlanDragAndDrop.rankIssues = function (issueKeys, sprintId, prevRankableId, nextRankableId) {
        if (GH.RapidBoard.State.isKanbanBoard()) {
            KanbanTransitionAndRank.transitionAndRankIssues(issueKeys, sprintId, prevRankableId, nextRankableId);

            //This return is to make sure we're returning the same thing as rankIssues. rankIssues returns
            //a deferred to represent the outcome of the sprint scope change dialog. If no dialog required, i.e. moving
            //to backlog or future sprint then it returns an already resolved deferred. If a dialog is shown then
            //whether the deferred object is resolved or rejected depends on whether the user clicks ok or cancel. In
            //our case there's no such confirmation step before doing the actual transition and/or rank, hence always
            //return a resolved deferred object.
            return $.Deferred().resolve();
        } else {
            return sprintRankIssues(issueKeys, sprintId, prevRankableId, nextRankableId);
        }
    };

    function sprintRankIssues(issueKeys, sprintId, prevRankableId, nextRankableId) {
        // put together the information about the move
        var issueMoveModel = IssueMoveController.calculateIssueMoveModel(issueKeys, sprintId);

        // kick off confirmation
        var confirmDeferred = IssueMoveController.confirmMove(issueMoveModel);

        // reload if the operation was canceled
        confirmDeferred.fail(function () {
            PlanController.reload();
        });

        // execute the rank if success
        confirmDeferred.done(function (issueMoveModel) {
            SprintController.triggerAnalyticsIssuesDnD(issueMoveModel);
            var issueKeysToMove = [];
            var issueKeysToRemove = [];

            // is this a complete move (so no possibility for done issues moving out of an active sprint)?
            // not possible if only a rank, neither possible when move between active sprints
            var isMove = IssueMoveController.isMoveBetweenActiveSprints(issueMoveModel) || IssueMoveController.isOnlyRankOperation(issueMoveModel);
            if (isMove) {
                issueKeysToMove = issueKeys;
            } else {
                // otherwise we have to distinguish between issues that are done/not done
                var change = issueMoveModel.changes[0];
                issueKeysToMove = change.issueKeys;
                issueKeysToRemove = change.doneIssueKeys;
            }

            // even before we know if ranking is successful, reorder issues in data model client side
            if (!_.isEmpty(issueKeysToMove)) {
                BacklogController.moveIssues(issueKeysToMove, sprintId, prevRankableId, nextRankableId);
            }

            // remove done issues
            if (!_.isEmpty(issueKeysToRemove)) {
                //  We know that all issues are from the same model...
                var model = BacklogModel.findModelWithIssue(issueKeysToRemove[0]);
                BacklogController.removeIssues(model, issueKeysToRemove);
            }

            // execute rank and sprint assignment, reload if that fails
            var deferreds = [];
            if (!_.isEmpty(issueKeysToMove)) {
                // move request
                deferreds.push(GH.RankController.sprintRankIssues(BacklogModel.getRankCustomFieldId(), issueKeysToMove, sprintId, nextRankableId, prevRankableId));
            } else {
                // place a null entry in the deferreds array, the moveResponse will be null
                deferreds.push($.Deferred().resolve(null));
            }
            if (!_.isEmpty(issueKeysToRemove)) {
                // remove request
                deferreds.push(GH.RankController.sprintRankIssues(BacklogModel.getRankCustomFieldId(), issueKeysToRemove, null, null, null));
            } else {
                // place a null entry in the deferreds array, the removeResponse will be null
                deferreds.push($.Deferred().resolve(null));
            }
            $.when.apply(null, deferreds).done(function (moveResponse, removeResponse) {
                SprintController.triggerAnalytics(issueMoveModel);
                // inspect move and remove responses to see if they have any RankResults with errors
                var errorRankResult = moveResponse && GH.RankController.getErrorRankResultFromResponse(moveResponse[0]);
                if (!errorRankResult) {
                    errorRankResult = removeResponse && GH.RankController.getErrorRankResultFromResponse(removeResponse[0]);
                }
                if (errorRankResult) {
                    // show RankResult errors
                    GH.Notification.showWarning(errorRankResult.errors[0]);
                    PlanController.reload();
                } else {
                    PlanDragAndDrop.refreshDetailsView(issueMoveModel);
                    IssueActions.showSprintRankSuccessMessage(issueMoveModel);
                }
                BacklogSelectionController.removeFromSelectionIssuesInDifferentModel(sprintId);
            }).fail(function () {
                // if ranking failed, reload model and redraw
                PlanController.reload();
            });
        });

        // also show a success message right away
        confirmDeferred.done(function () {
            PlanDragAndDrop.rankingComplete = true;
        });

        return confirmDeferred;
    }

    /**
     * Refresh the details view in response to an issue move,
     * only refreshing if the details view's issue changed scope.
     *
     * This primarily helps keep the issue tools menu's options relevant.
     *
     * @param issueMoveModel
     */
    PlanDragAndDrop.refreshDetailsView = function (issueMoveModel) {
        // Only re-render if the details view is open.
        if (!PlanController.isDetailsViewOpened()) {
            return;
        }

        // Only re-render if the operation changed the scope of an issue
        if (IssueMoveController.isOnlyRankOperation(issueMoveModel)) {
            return;
        }

        var selectedIssueKey = GH.DetailsView.selectedIssueKey;
        var allIssueKeys = IssueMoveController.getAllIssueKeys(issueMoveModel);
        if (_.contains(allIssueKeys, selectedIssueKey)) {
            GH.DetailsView.reload();
        }
    };

    //
    // Marker logic
    //

    /**
     * Called when marker dragging starts.
     */
    PlanDragAndDrop.markerDragStartHandler = function (event, ui) {

        var elem = $(ui.item),
            container = elem.parent(),
            containerOffset = container.offset(),
            markerHeight = elem.height(),
            $header = container.closest('.ghx-backlog-container').find('.js-sprint-header'),
            markerDragOverlay = $(GH.tpl.backlogview.renderDragOverlay({}));

        // following is the min top value for the overlay - as we don't want the overlay to spill into the header area
        // we have to do this because the overlay is appended to the body, and therefore not part of the backlog scrollable
        var minTop = $('#ghx-backlog').offset().top;

        var calcTop = function calcTop() {
            var top = $header.length ? $header.get(0).getBoundingClientRect().bottom : 0;
            return top < minTop ? minTop : top;
        };
        var calcHeight = function calcHeight(top) {
            return ui.helper.offset().top - top + markerHeight;
        };

        var top = calcTop();
        var height = calcHeight(top);

        markerDragOverlay.css({
            'top': top,
            'left': containerOffset.left + 1,
            'height': height,
            'width': container.outerWidth() - 2
        }).appendTo('body');

        PlanDragAndDrop.adjustMarkerDragOverlayHeight = function () {
            if (!this.options.axis || this.options.axis != "y") {
                this.helper[0].style.left = this.position.left + 'px';
            }
            if (!this.options.axis || this.options.axis != "x") {
                this.helper[0].style.top = this.position.top + 'px';
            }

            var top = calcTop();
            var height = calcHeight(top);
            markerDragOverlay[0].style.top = top + 'px';
            markerDragOverlay[0].style.height = height + 'px';
        };

        var marker = $(ui.helper);

        //    marker.parent().addClass('ghx-active-drag');
        //    header.parent().addClass('ghx-active-drag');
        elem.closest('.ghx-sprint-planned').addClass('ghx-active-drag');

        // Keep reference to element
        PlanDragAndDrop.currentDragElement = marker;
    };

    /**
     * Handles the update event of sortable markers.
     */
    PlanDragAndDrop.markerChangeHandler = function (event, ui) {
        // this is where the marker "is" (the placeholder's position relative to other sortables is kept up to date)
        var $placeholderMarker = $(ui.placeholder);

        // find the sprint id of the dragged marker
        var draggedSprintId = SprintView.getSprintIdForMarker(ui.item);
        if (!draggedSprintId) {
            return;
        }

        // find the sprint the marker is currently in as well as the previous issue (in that sprint)
        // null for backlog
        var currentSprintId = SprintView.getSprintIdForElement($placeholderMarker);
        var $previousRankable = PlanIssueListView.findPreviousVisibleIssueElementInList($placeholderMarker);
        var previousIssueKey = PlanIssueListView.getRankableId($previousRankable);

        // update the statistics of the dragged sprint
        PlanDragAndDrop.updateTemporarySprintStatistics(draggedSprintId, currentSprintId, previousIssueKey);
    };

    /**
     * Called when a marker drag operation stopped.
     */
    PlanDragAndDrop.markerDragStopHandler = function () {
        // clear out classes temporarily added during drag
        var issueListContainers = $('.ghx-backlog-container');
        issueListContainers.removeClass('ghx-active-drag ghx-overtaken');

        // remove the overlay
        $('.ghx-drag-overlay').remove();

        // remove reference to element
        PlanDragAndDrop.currentDragElement = null;
        BacklogView.adjustStickyHeader();
    };

    /**
     * Handles the update event of sortable markers.
     */
    PlanDragAndDrop.markerUpdateHandler = function (event, ui) {

        // do analytics before we even know if they are successful
        PlanDragAndDrop.analytics.sprints.trigger('footerdrag'); // SAFE

        // find the sprint id of the dragged marker
        var $draggedMarker = $(ui.item);
        var draggedSprintId = SprintView.getSprintIdForMarker($draggedMarker);
        if (!draggedSprintId) {
            return;
        }

        // find the sprint the marker is currently in as well as the previous issue (in that sprint)
        // null for backlog
        var currentSprintId = SprintView.getSprintIdForElement($draggedMarker);
        var $previousRankable = PlanIssueListView.findPreviousVisibleIssueElementInList($draggedMarker);
        var previousIssueKey = PlanIssueListView.getRankableId($previousRankable);

        // let the backlog controller handle the ranking
        PlanDragAndDrop.handleMarkerDragRanking(draggedSprintId, currentSprintId, previousIssueKey);
    };

    /**
     * Executes the marker move operation.
     *
     * Only visible issues are affected. Issues included will be moved to the bottom
     * of the current sprint, issues excluded will be ranked to the top of the next sprint/backlog
     */
    PlanDragAndDrop.handleMarkerDragRanking = function (draggedSprintId, currentSprintId, previousIssueKey) {
        // fetch the data required to rank
        var rankData = BacklogModel.calculateRankingDataForSprintMarkerMove(draggedSprintId, currentSprintId, previousIssueKey);

        // rank the issues
        if (rankData) {
            var deferred = PlanDragAndDrop.rankIssues(rankData.issueKeys, rankData.sprintId, rankData.prevRankableId, rankData.nextRankableId);

            // make sure the selection is still valid - it can't span multiple sprints
            deferred.done(function () {
                var selectionChanged = BacklogSelectionController.validateCurrentSelection();
                if (selectionChanged) {
                    // update the UI
                    BacklogView.updateIssueSelectionState(false);
                }
            });
        }

        // updated the current model in case the marker was placed before any issue. It is really hard to fix this case
        // in PlanDragAndDrop.changeHandler, so we fix it up here :/
        if (draggedSprintId != currentSprintId && !previousIssueKey) {
            // fetch the model for currentSprintId, as depending on where the marker lands this sprint might not be properly redrawn
            // (happens when marker is placed before first issue)
            var modelsToUpdate = [];

            if (currentSprintId) {
                modelsToUpdate.push(BacklogModel.getSprintModel(currentSprintId));
            } else {
                modelsToUpdate.push(BacklogModel.getBacklogModel2());
            }

            // if no rank operation too place, the dragged sprint won't be updated either, so fix this up too
            if (!rankData) {
                modelsToUpdate.push(BacklogModel.getSprintModel(draggedSprintId));
            }

            BacklogView.redrawChangedModels(modelsToUpdate);
        }
    };

    /**
     * Updates sprint statistics temporarily
     *
     * @param draggedSprintId the sprintId
     * @param currentSprintId the id of the sprint the marker is currently in
     * @param previousIssueKey the issue key of the first visible issue before the marker
     */
    PlanDragAndDrop.updateTemporarySprintStatistics = function (draggedSprintId, currentSprintId, previousIssueKey) {
        // fetch the model for the dragged sprint
        var sprintModel = BacklogModel.getSprintModel(draggedSprintId);
        if (!sprintModel) {
            return;
        }

        // calculate temporary data for the dragged marker/sprint
        var data = BacklogModel.calculateTemporaryDataForSprintWithMarker(draggedSprintId, currentSprintId, previousIssueKey);

        // set an overtaken class on affected models
        SprintView.markAsOvertaken(data.modelsData.otherAffectedModels);

        // update marker statistics
        SprintView.updateSprintStatisticForDraggedSprint(sprintModel, data.issueList);
    };

    /**
     * iterator for Underscore.js which selects elements by class name
     * @param className
     */
    PlanDragAndDrop._findShownElementByClass = function (className) {
        return function (elem) {
            var $elem = $(elem);
            return $elem.hasClass(className) && !$elem.is(":hidden");
        };
    };

    /**
     * iterator for Underscore.js which selects elements by class name
     * @param className
     */
    PlanDragAndDrop._findShownElement = function (elem) {
        var $elem = $(elem);
        return !$elem.is(":hidden");
    };

    PlanDragAndDrop.NotRankableHandler = {
        mouseXStart: 0,
        mouseYStart: 0,
        register: function register(selector) {
            $(document).delegate(selector, 'mousedown', PlanDragAndDrop.NotRankableHandler.mouseDownHandler);
        },
        registerOutUpHandlers: function registerOutUpHandlers(elem) {
            elem.bind('mouseout', PlanDragAndDrop.NotRankableHandler.mouseOutHandler);
            elem.bind('mouseup', PlanDragAndDrop.NotRankableHandler.mouseUpHandler);
        },
        unregisterOutUpHandlers: function unregisterOutUpHandlers(elem) {
            elem.unbind('mouseout', PlanDragAndDrop.NotRankableHandler.mouseOutHandler);
            elem.unbind('mouseup', PlanDragAndDrop.NotRankableHandler.mouseUpHandler);
        },
        mouseDownHandler: function mouseDownHandler() {
            var elem = $(this);
            PlanDragAndDrop.NotRankableHandler.registerOutUpHandlers(elem);
        },
        mouseOutHandler: function mouseOutHandler() {
            // unregister handlers
            var elem = $(this);
            PlanDragAndDrop.NotRankableHandler.unregisterOutUpHandlers(elem);

            PlanDragAndDrop.NotRankableHandler.showNotRankableWarning();
        },
        mouseUpHandler: function mouseUpHandler() {
            var elem = $(this);
            PlanDragAndDrop.NotRankableHandler.unregisterOutUpHandlers(elem);
        },
        showNotRankableWarning: function showNotRankableWarning() {
            var url = GH.RapidBoard.getRapidViewConfigurationUrl(BacklogController.rapidViewData.id, 'filter');
            var a = '<a href="' + url + '">';
            var endA = '</a>';

            var message = AJS.I18n.getText('gh.rapid.board.not.rankable.warning', a, endA);
            GH.Notification.showWarning(message);
        }
    };

    return PlanDragAndDrop;
});