(function (jq, _, BacklogModel, BacklogStatistics, SprintView, events) {
    'use strict';

    function getPlanModeData() {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var planModeData = {
            issueListStats: BacklogStatistics.getIssueListStats(backlogIssueList, false),
            sprintModels: BacklogModel.getSprintModels(),
            canCreateIssue: BacklogModel.canCreateIssue(),
            canManageSprints: BacklogModel.canManageSprints(),
            sprintIssueListStats: [],
            numActiveSprints: 0,
            numFutureSprints: 0
        };
        _.each(planModeData.sprintModels, function (sprintModel) {
            planModeData.sprintIssueListStats.push(SprintView.calculateSprintIssueListStats(sprintModel, sprintModel.getIssueList()));
            if (sprintModel.sprintData.state === 'ACTIVE') {
                planModeData.numActiveSprints++;
            }
            if (sprintModel.sprintData.state === 'FUTURE') {
                planModeData.numFutureSprints++;
            }
        });
        return planModeData;
    }

    /**
     * Update FillYourBacklog onboarding message
     */
    function updateFillYourBacklogMessage(planModeData) {
        var $container = jq('.ghx-onboarding-fillyourbacklog-group');
        var $backlogMessage = jq('.ghx-backlog-group .ghx-helper');
        if (planModeData.issueListStats.totalCount < 3 && planModeData.sprintModels.length === 0 && planModeData.canCreateIssue) {
            // render fill your backlog onboarding
            $container.html(GH.tpl.planonboarding.renderFillYourBacklogMessage());
            $backlogMessage.hide();
        } else if (planModeData.issueListStats.totalCount === 0 || planModeData.issueListStats.visibleCount === 0) {
            $backlogMessage.show();
            $container.empty();
        } else {
            $backlogMessage.hide();
            $container.empty();
        }
    }

    function updatePlanYourSprintMessage(planModeData) {
        var $containers, $description, sprintModel, sprintStats;
        if (planModeData.canManageSprints && planModeData.numActiveSprints === 0) {
            sprintModel = _.first(planModeData.sprintModels);
            sprintStats = _.first(planModeData.sprintIssueListStats);
            if (sprintModel && sprintStats) {
                if (sprintModel.sprintData.state === 'FUTURE' && sprintStats.totalCount === 0) {
                    $containers = jq('.js-sprint-container');
                    $description = $containers.filter('[data-sprint-id=' + sprintModel.sprintData.id + ']').find('.ghx-helper');
                    $description.html(GH.tpl.planonboarding.renderPlanYourSprintMessage());
                }
            }
        }
    }

    var PlanOnboarding = function PlanOnboarding() {
        events(this);
    };

    /**
     * Update StartSprint button and chaperone
     */
    PlanOnboarding.prototype.updateStartSprint = function (planModeData) {
        // ensure none of the start sprint buttons are primary
        jq('.js-sprint-start').removeClass('aui-button-primary');
        if (planModeData.canManageSprints && planModeData.numActiveSprints === 0) {
            // find first sprint eligible to be started
            var sprintModel = _.find(planModeData.sprintModels, function (sprintModel, index) {
                return sprintModel.sprintData.state === 'FUTURE' && planModeData.sprintIssueListStats[index].totalCount > 0;
            });
            if (sprintModel) {
                // make the associated start sprint button primary
                var sprintStartButtonSelector = '.js-sprint-container[data-sprint-id=' + sprintModel.sprintData.id + '] .js-sprint-start';
                jq(sprintStartButtonSelector).addClass('aui-button-primary');
            }
        }
    };

    PlanOnboarding.prototype.renderOnboarding = function () {
        var planModeData = getPlanModeData();
        updateFillYourBacklogMessage(planModeData);
        updatePlanYourSprintMessage(planModeData);
        this.updateCreateSprint(planModeData);
        this.updateStartSprint(planModeData);
        this.refresh();
    };

    /**
     * Refresh elements (usually due to change in layout).
     */
    PlanOnboarding.prototype.refresh = function () {
        this.trigger('refresh');
    };

    function createSprintShouldBePrimary(planModeData) {
        return planModeData.canManageSprints && planModeData.numFutureSprints === 0 && planModeData.numActiveSprints === 0 && planModeData.issueListStats.totalCount >= 3;
    }

    /**
     * Update CreateSprint button and chaperone
     */
    PlanOnboarding.prototype.updateCreateSprint = function (planModeData) {
        var $button = jq('.js-add-sprint');
        if (createSprintShouldBePrimary(planModeData)) {
            $button.addClass('aui-button-primary');
        } else {
            $button.removeClass('aui-button-primary');
        }
    };

    GH.PlanOnboarding = PlanOnboarding;
    GH.planOnboarding = new PlanOnboarding();
})(AJS.$, _, GH.BacklogModel, GH.BacklogStatistics, GH.SprintView, GH.Events);