/* globals
 * GH.RapidBoard, GH.EpicController, GH.VersionController
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-url-state
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-controls
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 */
define('jira-agile/rapid/ui/plan/plan-url-state', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');

    /**
     * Manages the url state for the plan tab
     */
    var PlanUrlState = {};

    PlanUrlState.self = "PlanUrlState";
    /**
     * Is this state applicable for the given normalized state
     * @param normalized
     */
    PlanUrlState.isApplicable = function (normalized) {
        return normalized['mode'] == 'plan';
    };

    /**
     * Normalizes the url state for the plan specific values.
     * @param urlParams the parsed url parameters
     * @param normalized normalized parameters
     */
    PlanUrlState.getNormalizedFromUrl = function (urlParams, normalized) {
        // backlog/active sprint issue
        normalized['plan.selectedIssue'] = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['selectedIssue'], null);

        // quick filter
        normalized['quickFilter'] = GH.RapidBoard.UrlState.normalizeNumberArrayParam(urlParams['quickFilter']); // empty by default

        normalized['versions'] = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['versions'], '') === 'visible'; // hidden

        // epic tab
        normalized['epics'] = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['epics'], '') === 'visible'; // hidden by default

        // selected epic
        normalized['plan.selectedEpic'] = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['selectedEpic'], null);

        normalized['plan.selectedVersion'] = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['selectedVersion'], null);
    };

    /**
     * Transforms the normalized url parameters into actual url parameters
     * @param normalized the normalized params
     * @param urlParams the url params
     */
    PlanUrlState.toUrl = function (normalized, urlParams) {
        // backlog/active sprint issue
        if (!_.isNull(normalized['plan.selectedIssue'])) {
            urlParams['selectedIssue'] = normalized['plan.selectedIssue'];
        }

        // filters
        if (normalized['quickFilter'].length > 0) {
            urlParams['quickFilter'] = normalized['quickFilter'];
        }

        if (normalized['versions']) {
            urlParams['versions'] = 'visible';
        }

        // epic tab
        if (normalized['epics']) {
            urlParams['epics'] = 'visible';
        }

        // selected epic
        if (!_.isNull(normalized['plan.selectedEpic'])) {
            urlParams['selectedEpic'] = normalized['plan.selectedEpic'];
        }

        if (!_.isNull(normalized['plan.selectedVersion'])) {
            urlParams['selectedVersion'] = normalized['plan.selectedVersion'];
        }
    };

    /**
     * Get the normalized state from the app
     * @param normalized the normalized state to put the values into
     */
    PlanUrlState.getNormalizedFromInternal = function (normalized) {
        PlanController.init();

        // backlog/active sprint issue
        normalized['plan.selectedIssue'] = GH.BacklogSelectionController.getSelectedIssueKey();

        // quick filter
        normalized['quickFilter'] = PlanControls.quickfilters.getActiveQuickFilters();

        normalized['versions'] = PlanController.isVersionsColumnVisiblePersisted();

        // epic tab
        normalized['epics'] = PlanController.isEpicsColumnVisiblePersisted();

        // selected epic
        normalized['plan.selectedEpic'] = GH.EpicController.getFilteredEpicKey();

        normalized['plan.selectedVersion'] = GH.VersionController.getFilteredVersionId();
    };

    /**
     * Update the app with the values stored in the normalized state
     * @param normalized the normalized state to update the values from
     */
    PlanUrlState.updateInternalFromNormalized = function (normalized) {
        PlanController.init();
        GH.BacklogSelectionController.setIssueFromUrlState(normalized['plan.selectedIssue']);
        PlanControls.quickfilters.setActiveQuickFilters(normalized['quickFilter']);
        PlanController.setVersionsColumnVisible(normalized['versions']);
        PlanController.setEpicsColumnVisible(normalized['epics']);
        GH.EpicController.setFilteredEpicKey(normalized['plan.selectedEpic'], true);
        GH.VersionController.setFilteredVersionId(normalized['plan.selectedVersion'], true);
    };

    return PlanUrlState;
});