/**
 * Controller for the grid data model
 */

/**
 * Data held by this model
 */
/*
GH.GridDataModel.data = {
    columns: [],
    columnIssueMapping: {},
    swimlanes: [],
    swimlaneIssueMapping: {},
    issues: {},
    order: [],
    anyColumnBusted: false,
};
*/
define('jira-agile/rapid/ui/work/grid-data-model', ["jira-agile/rapid/logger", "underscore", "jira-agile/rapid/ui/component/extra-fields/ExtraFieldsHelper", "jira-agile/rapid/ui/work/issue-list-util"], function (Logger, _, ExtraFieldsHelper, IssueListUtil) {
    "use strict";
    /**
     * Grid data model implemenation
     *
     * A grid model holds rows of content
     */

    var GridDataModel = function GridDataModel(columns, columnIssueMapping, swimlanes, swimlaneIssueMapping, issues, order, variables, swimlaneStrategy) {
        var data = this.data = {};
        data.columns = columns;
        data.columnIssueMapping = columnIssueMapping;
        data.swimlanes = swimlanes;
        data.swimlaneIssueMapping = swimlaneIssueMapping;
        data.issues = issues;
        data.order = order;
        data.variables = variables;
        data.swimlaneStrategy = swimlaneStrategy;

        // cache common needed structures
        this.buildCache();

        // preprocess the issues
        this.preprocessIssues();
    };

    GridDataModel.prototype.isModelDifferent = function (otherModel) {
        // compare order
        if (!_.isEqual(this.data.order, otherModel.data.order)) {
            Logger.log('GridDataModel comparison: The order differs');
            return true;
        }

        // compare issues
        if (!_.isEqual(this.data.issues, otherModel.data.issues)) {
            Logger.log('GridDataModel comparison: The issues details differ');
            return true;
        }

        // compare swimlanes
        if (!_.isEqual(this.data.swimlaneIssueMapping, otherModel.data.swimlaneIssueMapping)) {
            Logger.log('GridDataModel comparison: The swimlanes differ');
            return true;
        }

        // compare columns
        if (!_.isEqual(this.data.columnIssueMapping, otherModel.data.columnIssueMapping)) {
            Logger.log('GridDataModel comparison: The columns differ');
            return true;
        }

        // could check for swimlane/column config here as well, but leaving it out for now (we currently don't
        // properly handle rapid view configuration changes anywhere on the board

        // order + issues + grid is still the same, the two models are therefore equal
        Logger.log('GridDataModel comparison: Everything is equal');
        return false;
    };

    /**
     * Builds a cache for the given model, thus turns the several flat structures into a tree
     * structure with rows, column, columns inside rows, to speed up model querying
     */
    GridDataModel.prototype.buildCache = function () {
        var data = this.data;
        var cache = this.cache = {};
        cache.bySwimlaneThenColumn = [];
        var self = this;
        _.each(data.swimlanes, function (swimlane) {
            var swimlaneKeys = data.swimlaneIssueMapping[swimlane.id];
            cache.bySwimlaneThenColumn[swimlane.id] = [];
            _.each(data.columns, function (column) {
                var columnKeys = data.columnIssueMapping[column.id];
                // TODO: if column Keys aren't sorted according to order we are in trouble...
                var a = self.fastIntersect(columnKeys, swimlaneKeys);
                // apply the order to the issues
                a = self.fastIntersect(data.order, a);
                cache.bySwimlaneThenColumn[swimlane.id][column.id] = a;
            });
        });
    };

    /**
     * _.intersect is really slow on IE, order a and b and apply a faster algorithm
     *
     * The ordering of the result will be according to the order in a.
     *
     * @param a the first array
     * @param b the second array
     */
    GridDataModel.prototype.fastIntersect = function (a, b) {
        var aLen = a.length;
        var bLen = b.length;

        // sort both
        var aSorted = a.slice(0).sort(); // first copy, then sort
        var bSorted = b.slice(0).sort(); // first copy, then sort

        var aAndB = [];
        var aIndex = 0,
            bIndex = 0;
        while (aIndex < aLen && bIndex < bLen) {
            var key = aSorted[aIndex];
            var index = bSorted.indexOf(key, bIndex);
            if (index > -1) {
                aAndB.push(key);
                bIndex = index + 1;
            }
            aIndex++;
        }

        // we need to keep the ordering of a
        var result = [];
        for (var i = 0; i < aLen; i++) {
            if (aAndB.indexOf(a[i]) > -1) {
                result.push(a[i]);
            }
        }

        return result;
    };

    /**
     * Preprocesses issues
     */
    GridDataModel.prototype.preprocessIssues = function () {
        var self = this;
        _.each(this.data.issues, function (issue) {
            self.preprocessIssue(issue);
        });
    };

    /**
     * Precalculates some issue information
     * @param issueData
     */
    GridDataModel.prototype.preprocessIssue = function (issueData) {
        // time in column
        this.computeTimeInColumnForIssue(issueData);
        // mark sanitised html for soy
        ExtraFieldsHelper.prepareExtraFields(issueData.extraFields);
    };

    /**
     * Compute time spent in current column. The server gives us the total duration in milliseconds,
     * at the moment we hardcode that to days (24h). Rounding is always down.
     */
    GridDataModel.prototype.computeTimeInColumnForIssue = function (issueData) {
        var day = 86400000; // 1000*60*60*24;
        if (issueData.timeInColumn) {
            issueData.timeInColumn.days = Math.floor((issueData.timeInColumn.durationPreviously + // time previously spent in column
            new Date().getTime() - issueData.timeInColumn.enteredStatus) / // + time spent in current status
            day);
        }
    };

    /**
     * Updates the data for a single issue
     */
    GridDataModel.prototype.updateIssueData = function (issueData) {
        this.preprocessIssue(issueData);

        // replace issue
        this.data.issues[issueData.key] = issueData;

        // todo: could calculate whether the issue changed column/swimlane
    };

    /**
     * Update the order
     */
    GridDataModel.prototype.updateOrder = function (order) {
        this.data.order = order;
        this.buildCache();
    };

    /**
     * Compute the issue count data for columns, swimlanes and totals
     * Also fills out required data for issue parents
     */
    GridDataModel.prototype.getGroupData = function () {
        // gather all the lists we want to process
        var lists = [];
        var self = this;
        _.each(self.data.swimlanes, function (swimlane) {
            _.each(self.data.columns, function (column) {
                lists.push(self.getIssuesDataForCell(swimlane.id, column.id));
            });
        });

        // calculate the group data
        var data = IssueListUtil.calculateGroupData(lists);

        // should fake parents be drawn?
        data.drawFakeParents = !this.isParentChildStrategy();
        return data;
    };

    /**
     * Gets the issues for a given cell
     */
    GridDataModel.prototype.getIssuesDataForCell = function (swimlaneId, columnId) {
        var columnsInSwimlane = this.cache.bySwimlaneThenColumn[swimlaneId];
        if (!columnsInSwimlane) {
            return false;
        }
        var issueKeys = columnsInSwimlane[columnId];
        if (!issueKeys) {
            return false;
        }
        return this.getIssueDataByKeys(issueKeys);
    };

    /**
     * Returns a list of issueData given the corresponding issues' keys
     */
    GridDataModel.prototype.getIssueDataByKeys = function (issueKeys) {
        var issueData = [];
        var issues = this.data.issues;
        _.each(issueKeys, function (issueKey) {
            var data = issues[issueKey];
            if (data) {
                issueData.push(data);
            }
        });
        return issueData;
    };

    // Getters

    /**
     * Returns the ordered issues in a lookup array by swimlaneId then columnId
     */
    GridDataModel.prototype.getCells = function () {
        return this.cache.bySwimlaneThenColumn;
    };

    /**
     * Get all issues
     */
    GridDataModel.prototype.getIssues = function () {
        return this.data.issues;
    };

    /**
     * Get all columns
     */
    GridDataModel.prototype.getColumns = function () {
        return this.data.columns;
    };

    /**
     * Get all swimlanes
     */
    GridDataModel.prototype.getSwimlanes = function () {
        return this.data.swimlanes;
    };

    /**
     * Get the order data
     */
    GridDataModel.prototype.getOrder = function () {
        return this.data.order;
    };

    // Getters with params

    /**
     * Get a swimlane given its id
     */
    GridDataModel.prototype.getSwimlaneById = function (swimlaneId) {
        return _.find(this.data.swimlanes, function (swimlane) {
            return swimlane.id == swimlaneId;
        });
    };

    /**
     * Get a column given its id
     */
    GridDataModel.prototype.getColumnById = function (columnId) {
        return _.find(this.data.columns, function (column) {
            return column.id == columnId;
        });
    };

    GridDataModel.prototype.getFirstColumn = function () {
        return _.first(this.data.columns);
    };

    /**
     * Get the swimlane id a given issue is currently in
     */
    GridDataModel.prototype.getSwimlaneIdByIssueKey = function (issueKey) {
        var foundId = false;
        _.each(this.data.swimlaneIssueMapping, function (swimlaneIssues, swimlaneId) {
            if (_.indexOf(swimlaneIssues, issueKey) > -1) {
                foundId = parseInt(swimlaneId, 10);
            }
        });
        return foundId;
    };

    /**
     * Get the column id a given issue is currently in
     */
    GridDataModel.prototype.getColumnIdByIssueKey = function (issueKey) {
        var foundId = false;
        _.each(this.data.columnIssueMapping, function (columnIssues, columnId) {
            if (_.indexOf(columnIssues, issueKey) > -1) {
                foundId = parseInt(columnId, 10);
            }
        });
        return foundId;
    };

    /**
     * Get the issue data for a given issue key
     */
    GridDataModel.prototype.getIssueDataByKey = function (issueKey) {
        return this.data.issues[issueKey];
    };

    /**
     * Get the issue data for a given issue id
     */
    /* global _ */
    GridDataModel.prototype.getIssueDataById = function (issueId) {
        return _.findWhere(this.data.issues, { id: issueId });
    };

    /**
     * Get the issue id for a given issue key
     */
    GridDataModel.prototype.getIssueIdForKey = function (issueKey) {
        var data = this.getIssueDataByKey(issueKey);
        return data ? data.id : false;
    };

    // Counts and checks

    GridDataModel.prototype.isIssueValid = function (issueKey) {
        return this.data.issues[issueKey] ? true : false;
    };

    GridDataModel.prototype.hasSwimlanes = function () {
        var self = this;
        var parentStrategy = self.isParentChildStrategy();
        return _.any(self.getSwimlanes(), function (swimlane) {
            // in case of parent strategy we even show swimlanes if they are empty
            return (parentStrategy || self.getSwimlaneIssueCount(swimlane.id) > 0) && !swimlane.defaultSwimlane;
        });
    };

    GridDataModel.prototype.getSwimlaneIssueCount = function (swimlaneId) {
        return this.data.swimlaneIssueMapping[swimlaneId].length;
    };

    GridDataModel.prototype.getColumnIssueCount = function (columnId) {
        return this.data.columnIssueMapping[columnId].length;
    };

    /**
     * Is the model representing a parent child swimlane strategy
     */
    GridDataModel.prototype.isParentChildStrategy = function () {
        return this.data.variables.swimlanesType == 'parentChild';
    };

    /**
     * Get a swimlane given the parent kye
     */
    GridDataModel.prototype.getParentSwimlaneByIssueKey = function (issueKey) {
        if (!this.isParentChildStrategy()) {
            return false;
        }

        return _.find(this.data.swimlanes, function (lane) {
            if (lane.parentKey == issueKey) {
                return true;
            }
        }) || false;
    };

    // Position based navigation


    /**
     * Return position data for a given issue
     *
     * Note: this is an internal function that should not be called outside of this controller
     */
    GridDataModel.prototype._getIssuePositionAndDataByKey = function (issueKey) {
        // fetch issue data first
        var issueData = this.data.issues[issueKey];
        if (!issueData) {
            return false;
        }

        // check whether this is a swimlane issue
        var swimlane = false;
        if (this.isParentChildStrategy()) {
            _.each(this.data.swimlanes, function (lane) {
                if (lane.parentKey == issueKey) {
                    swimlane = lane;
                }
            });
        }
        if (swimlane) {
            return {
                isSwimlane: true,
                swimlaneId: swimlane.id,
                columnId: _.first(this.getColumns()).id, // fix to first column
                issueIndex: -1,
                issue: issueData
            };
        }

        // fetch swimlane id
        var swimlaneId = this.getSwimlaneIdByIssueKey(issueKey);
        // a released issue is not in any swim lane
        if (!swimlaneId) {
            return false;
        }

        // fetch column id
        var columnId = this.getColumnIdByIssueKey(issueKey);
        // an issue in kanban backlog has no columnId
        if (!columnId) {
            return false;
        }

        // find the index inside the cell
        var index = -1;
        if (swimlaneId && columnId) {
            var cell = this.cache.bySwimlaneThenColumn[swimlaneId][columnId];
            index = _.indexOf(cell, issueKey);
        }

        // return the data
        return {
            swimlaneId: swimlaneId,
            columnId: columnId,
            issueIndex: index,
            issue: issueData
        };
    };

    /**
     * Get the index of an issue within a column
     */
    GridDataModel.prototype.getIssueIndexByKey = function (issueKey) {
        var data = this._getIssuePositionAndDataByKey(issueKey);
        return data ? data.issueIndex : false;
    };

    /**
     * Get the id of the following issue.
     */
    GridDataModel.prototype.getNextIssueKey = function (issueKey, currentColumn) {
        var data = this._getIssuePositionAndDataByKey(issueKey);
        if (!data) {
            return false;
        }

        // overwrite the column id if we got one
        if (currentColumn) {
            data.columnId = currentColumn;
        }

        var self = this;
        var cells = self.getCells();
        var swimlaneFound = false;
        var nextIssueKey = false;

        _.find(self.getSwimlanes(), function (swimlane) {
            var index;
            if (swimlane.id == data.swimlaneId) {
                swimlaneFound = true;
                index = data.issueIndex + 1; // works even for the isSwimlane data object
            } else {
                // skip if not yet at swimlane of current issue
                if (!swimlaneFound) {
                    return false;
                }

                // check whether this is a parent swimlane
                if (swimlane.parentKey) {
                    nextIssueKey = swimlane.parentKey;
                    return true;
                } else {
                    // otherwise check the first issue
                    index = 0;
                }
            }

            // ensure we only check opened swimlanes
            var hasSwimlanes = self.hasSwimlanes();
            if (hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id)) {
                return false;
            }

            // return issue at index
            nextIssueKey = cells[swimlane.id][data.columnId][index];
            return !!nextIssueKey;
        });
        return nextIssueKey || false;
    };

    /**
     * Get the id of the previous issue.
     */
    GridDataModel.prototype.getPreviousIssueKey = function (issueKey, currentColumn) {
        var data = this._getIssuePositionAndDataByKey(issueKey);
        if (!data) {
            return false;
        }

        // overwrite the column id if we got one
        if (currentColumn) {
            data.columnId = currentColumn;
        }

        var self = this;
        var cells = self.getCells();
        var swimlaneFound = false;
        var previousIssueKey = false;
        var reversedSwimlanes = self.getSwimlanes().slice(0).reverse();
        _.find(reversedSwimlanes, function (swimlane) {
            // skip we we aren't yet at the current swimlane
            if (swimlane.id != data.swimlaneId && !swimlaneFound) {
                return false;
            }
            swimlaneFound = true;

            // we need to handle the current lane different than the other lanes
            if (swimlane.id == data.swimlaneId) {

                // check whether the swimlane itself is selected, if so continue with previous swimlane
                if (swimlane.parentKey == issueKey) {
                    return false;
                }

                // check whether we are at the first issue and this is a parent swimlane, select its key in that case
                if (data.issueIndex == 0 && swimlane.parentKey) {
                    previousIssueKey = swimlane.parentKey;
                    return true;
                } else {
                    // take previous issue
                    previousIssueKey = cells[swimlane.id][data.columnId][data.issueIndex - 1];
                    return !!previousIssueKey;
                }
            }

            // select last issue if swimlane not collapsed, select swimlane if parent issue, skip otherwise
            var isCollapsed = self.hasSwimlanes() && GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id);
            if (!isCollapsed && cells[swimlane.id][data.columnId].length > 0) {
                previousIssueKey = _.last(cells[swimlane.id][data.columnId]);
                return !!previousIssueKey;
            } else if (swimlane.parentKey) {
                previousIssueKey = swimlane.parentKey;
                return true;
            } else {
                return false;
            }
        });
        return previousIssueKey || false;
    };

    /**
     * Get the key for the issue in the given column at given index.
     *
     * If the index does not exist then the last issue in the column is returned.
     * If no issue are available in the given column, false is returned
     */
    GridDataModel.prototype.getIssueKeyForColumnAtIndex = function (swimlaneId, columnId, issueIndex) {
        var cell = this.cache.bySwimlaneThenColumn[swimlaneId][columnId];
        if (_.isEmpty(cell)) {
            return false;
        }

        if (issueIndex < 0) {
            return cell[0];
        } else if (issueIndex >= cell.length) {
            return _.last(cell);
            // otherwise we want to get the issue at that index.
        } else {
            return cell[issueIndex];
        }
    };

    /**
     * Get the first issue in the view, starting with the first column
     */
    GridDataModel.prototype.getFirstIssueKey = function () {
        var swimlanes = this.getSwimlanes();
        var columns = this.getColumns();
        var cells = this.getCells();

        var hasSwimlanes = this.hasSwimlanes();
        var firstIssueKey = false;
        _.find(columns, function (column) {
            _.find(swimlanes, function (swimlane) {
                // skip if swimlane hidden
                if (hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id)) {
                    return false;
                }

                // fetch first key in cell
                var cell = cells[swimlane.id][column.id];
                if (_.isEmpty(cell)) {
                    return false;
                }

                firstIssueKey = _.first(cell);
                return true;
            });
            return firstIssueKey;
        });
        return firstIssueKey;
    };

    /**
     * Get the last issue in the view (beginning in the first column!)
     */
    GridDataModel.prototype.getLastIssueKey = function () {
        var reversedSwimlanes = this.getSwimlanes().slice().reverse();
        var columns = this.getColumns();
        var cells = this.getCells();

        var hasSwimlanes = this.hasSwimlanes();
        var lastIssueKey = false;
        _.find(columns, function (column) {
            _.find(reversedSwimlanes, function (swimlane) {
                // skip if swimlane hidden
                if (hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id)) {
                    return false;
                }

                // fetch first key in cell
                var cell = cells[swimlane.id][column.id];
                if (_.isEmpty(cell)) {
                    return false;
                }

                lastIssueKey = _.last(cell);
                return true;
            });
            return lastIssueKey;
        });
        return lastIssueKey;
    };

    /**
     * Returns the issue to the left of the issue represented in issueData, increasing offset to savedIndex if higher than
     * issueData's. returns undefined when no such issue exists
     * @param issueData structure of the form returned by getIssueDataByKey
     */
    GridDataModel.prototype.getIssueKeyInPreviousColumn = function (issueKey, savedIndex) {
        var data = this._getIssuePositionAndDataByKey(issueKey);
        if (!data) {
            return false;
        }

        // if this is a swimlane, ignore and return
        if (data.isSwimlane) {
            return false;
        }

        // check previous column until an issue is found
        var previousColumn = this._getPreviousColumn(data.columnId);
        while (previousColumn) {
            var previousIssueKey = this.getIssueKeyForColumnAtIndex(data.swimlaneId, previousColumn.id, savedIndex);
            if (previousIssueKey) {
                return previousIssueKey;
            }
            previousColumn = this._getPreviousColumn(previousColumn.id);
        }
        return false;
    };

    /**
     * Get the previous column given a column id
     */
    GridDataModel.prototype._getPreviousColumn = function (columnId) {
        var columns = this.getColumns();
        for (var i = columns.length - 1; i > 0; i--) {
            if (columns[i].id == columnId) {
                return columns[i - 1];
            }
        }
        return false;
    };

    /**
     * Returns the issue to the right of the issue represented in issueData, increasing offset to savedIndex if higher than
     * issueData's. returns undefined when no such issue exists
     * @param issueData structure of the form returned by getIssueDataByKey
     */
    GridDataModel.prototype.getIssueKeyInNextColumn = function (issueKey, savedIndex) {
        var data = this._getIssuePositionAndDataByKey(issueKey);
        if (!data) {
            return false;
        }

        // if this is a swimlane, ignore and return
        if (data.isSwimlane) {
            return false;
        }

        // check next column until an issue is found
        var nextColumn = this._getNextColumn(data.columnId);
        while (nextColumn) {
            var nextIssueKey = this.getIssueKeyForColumnAtIndex(data.swimlaneId, nextColumn.id, savedIndex);
            if (nextIssueKey) {
                return nextIssueKey;
            }
            nextColumn = this._getNextColumn(nextColumn.id);
        }
        return false;
    };

    /**
     * Get the next column given a column id
     */
    GridDataModel.prototype._getNextColumn = function (columnId) {
        var columns = this.getColumns();
        for (var i = 0; i < columns.length - 1; i++) {
            if (columns[i].id == columnId) {
                return columns[i + 1];
            }
        }
        return false;
    };

    // Issue position and selection of next issue in same spot

    /**
     * get the position (as a combination of swimlane, column and index within that column), for
     * the issue with key issueKey
     */
    GridDataModel.prototype.getIssuePositionByKey = function (issueKey) {
        var issueData = this._getIssuePositionAndDataByKey(issueKey);
        if (!issueData) {
            return undefined;
        }
        return {
            swimlaneId: issueData.swimlaneId,
            columnId: issueData.columnId,
            position: issueData.issueIndex
        };
    };

    /**
     * For a given location, find the issue at, or nearest to it (dropping to the top of the following
     * swimlane if we've overshot the specified swimlane/column combo).
     * @location the specified location, of the form {swimlaneId, columnId, position}
     */
    GridDataModel.prototype.getIssueKeyAtPosition = function (location) {

        // if location doesn't have the shape we expect return undefined
        if (!location || location.swimlaneId === undefined || location.columnId === undefined || location.position === undefined) {
            return undefined;
        }
        var selectedSwimlane, selectedColumn, selectedIssueKey, nextSwimlane;

        // select the swimlane with location.swimlaneId
        var swimlanes = this.getSwimlanes();
        for (var i = 0, l = swimlanes.length; i < l; i++) {
            if (swimlanes[i].id == location.swimlaneId) {
                selectedSwimlane = swimlanes[i];
                if (i + 1 < l) {
                    nextSwimlane = swimlanes[i + 1];
                }
                break;
            }
        }
        if (!selectedSwimlane) {
            return undefined;
        }

        // get the column for the given id
        selectedColumn = this.getColumnById(location.columnId);
        if (!selectedColumn) {
            return undefined;
        }

        // fetch the cell
        var cell = this.cache.bySwimlaneThenColumn[location.swimlaneId][location.columnId];

        // select the issue from the selected column
        selectedIssueKey = cell[location.position];

        // if we got an issue, return it
        if (selectedIssueKey) {
            return selectedIssueKey;

            // otherwise probe next swimlane
        } else if (nextSwimlane) {
            // if we didn't get an issue, then we want to grab the top issue from the next swimlane, so recurse
            return this.getIssueKeyAtPosition({
                swimlaneId: nextSwimlane.id,
                columnId: location.columnId,
                position: 0
            });

            // last issue in current swimlane or give up
        } else {
            // select last issue from given cell
            return _.last(cell);
        }
    };

    GridDataModel.prototype.moveIssueToColumn = function (issueKey, oldColumnId, newColumnId) {
        this.moveIssuesToColumn([issueKey], oldColumnId, newColumnId);
    };

    GridDataModel.prototype.moveIssuesToColumn = function (issueKeys, oldColumnId, newColumnId) {
        var columnIssueMapping = this.data.columnIssueMapping;
        if (!columnIssueMapping[oldColumnId] || !columnIssueMapping[newColumnId]) {
            return false;
        }

        issueKeys.forEach(function (issueKey) {
            // remove in old and add to new
            columnIssueMapping[oldColumnId] = _.without(columnIssueMapping[oldColumnId], issueKey);
            columnIssueMapping[newColumnId].push(issueKey);
        });

        // update the cache
        this.buildCache();
    };

    /**
     * Removes the epic from the issues with the given issue keys.
     *
     * @param issueKeys
     * @returns {Array} of issues who have been removed from their associated epic
     */
    GridDataModel.prototype.removeEpicFromIssues = function (issueKeys) {
        var issues = [];
        var issuesData = this.getIssueDataByKeys(issueKeys);
        _.each(issuesData, function (issueData) {
            if (issueData.epic) {
                delete issueData.epic;
                issues.push(issueData);
            }
        });
        return issues;
    };

    GridDataModel.prototype.getSwimlaneStrategy = function () {
        return this.data.swimlaneStrategy;
    };

    return GridDataModel;
});

AJS.namespace('GH.GridDataModel', null, require('jira-agile/rapid/ui/work/grid-data-model'));