define('jira-agile/rapid/ui/work/swimlane-view', [
    "jira-agile/rapid/ui/work/ranking-model",
    "jira-agile/rapid/ui/work/grid-data-controller",
    "jira-agile/rapid/user-data",
    "jira-agile/rapid/layout",
    "jira-agile/rapid/ui/work/swimlane-view-text",
    "jira-agile/rapid/ui/work/ActiveSprintEmptyView",
    "jira/util/formatter",
    "jquery",
    'underscore'
], function(RankingModel, GridDataController, UserData, Layout, SwimlaneViewText, ActiveSprintEmptyView, formatter, $, _) {
    "use strict";

    /**
     * Swimlane view
     */
    const SwimlaneView = {};

    /** Is the swimlane view currently visible? */
    SwimlaneView.visible = false;

    /** Holds the id of the detail view column */
    SwimlaneView.detailViewColumnId = false;

    /** Rapid View data */
    SwimlaneView.rapidViewData = {};

    /**
     * Called upon page load. Registers events this component is interested in
     */
    SwimlaneView.init = function () {
        // Stalker bar
        GH.SwimlaneStalker.init();

        // swimlane toggles
        $(document).delegate('#ghx-pool .ghx-swimlane-header .js-expander', 'click', SwimlaneView.toggleCollapsedHandler);
        $(document).delegate('#ghx-pool .ghx-swimlane-header .js-expander', 'keydown', SwimlaneView.toggleCollapsedHandler);

        // expand/collapse all in tools menu
        $(document).delegate('.js-view-action-expand-all', 'click', SwimlaneView.expandAll);
        $(document).delegate('.js-view-action-collapse-all', 'click', SwimlaneView.collapseAll);

        // release version
        $(document).delegate('#ghx-release', 'click', SwimlaneView.openReleaseVersionDialog);
        $(GH).bind(GH.Dialogs.ReleaseVersionDialog.EVENT_VERSION_RELEASED, GH.WorkController.reload);

        // move to final column
        $(document).delegate('button.js-sync', 'click', function () {
            // track that a user has clicked the move to done button.
            GH.WorkController.analytics.trigger('movetodonebutton.clicked');

            // Need to pull out the issue key from the parent div
            var parentKey = $(this).closest('.ghx-swimlane-header').attr('data-issue-key');
            var parentIssue = GridDataController.getModel().getIssueDataByKey(parentKey);
            GH.WorkController.resolveParentTask(parentIssue);
        });
    };

    /**
     * Called to show/render the swimlane view
     */
    SwimlaneView.show = function () {
        SwimlaneView.visible = true;
        SwimlaneView.renderSwimlanes();
    };

    /**
     * Called before hiding the swimlane view
     */
    SwimlaneView.hide = function () {
        SwimlaneView.visible = false;
    };

    /**
     * Set the current view data
     */
    SwimlaneView.setRapidView = function (rapidViewData) {
        SwimlaneView.rapidViewData = rapidViewData;
    };

    /**
     * Set the current view CONFIG data (DIFFERENT!)
     */
    SwimlaneView.setRapidViewConfig = function (rapidViewConfig) {
        SwimlaneView.rapidViewConfig = rapidViewConfig;
    };

    /**
     * Renders the swim lanes into the pool
     */
    SwimlaneView.renderSwimlanes = function () {
        var container = $('#ghx-pool');
        // don't render if we don't have the element to render into
        if (container.length < 1) {
            return;
        }

        // initialise the header offset right before the columns are drawn. It is correct at this point,
        // provided that things are being rendered in the right order, and _much_ cheaper to do since the
        // DOM is pretty small still.
        // This currently depends on OperationsBar to do have done its rendering.
//    GH.SwimlaneStalker.initHeadOffset();
        GH.SwimlaneStalker.poolStalker();

        container.hide();

        // data model to render
        var model = GridDataController.getModel();

        // add a class that defines whether more than one swimlane is displayed. Required to show/hide the swimlane name
        var hasSwimlanes = model.hasSwimlanes();
        if (hasSwimlanes) { // Adds the class for 1+ non-default swimlane
            container.addClass('ghx-has-swimlanes');
        } else {
            container.removeClass('ghx-has-swimlanes');
        }

        // set the parent/child strategy class
        var parentChildStrategy = model.isParentChildStrategy();
        if (parentChildStrategy) {
            container.addClass('ghx-parent-child');
        } else {
            container.removeClass('ghx-parent-child');
        }

        // render each column headers
        var columns = model.getColumns();
        var html = [GH.tpl.rapid.swimlane.renderColumnsHeader({
            columns: columns,
            statistics: GridDataController.getStatistics()
        })];

        // get the calculated display issue keys to the width we got available
        var displayIssueKeys = SwimlaneView.getCalculatedDisplayIssueKeys();

        // fetch data needed for the swimlanes
        var swimlanes = model.getSwimlanes();
        var cells = model.getCells();
        var issues = model.getIssues();
        var groupData = model.getGroupData();

        // now render all swimlanes, and keep them in an array
        var visibleSwimlaneIndex = 0;
        _.each(swimlanes, function (swimlane) {
            var issueCount = model.getSwimlaneIssueCount(swimlane.id);
            // show swimlane if we are in parentChildStrategy and its not the default lane or there are issues, or the
            // swimlane exceeded the maximum of issues
            if ((parentChildStrategy && !swimlane.defaultSwimlane) ||
                issueCount > 0) {
                // swimlane can only be collapsed if we "have a swimlane", as well as it being collapsed
                var isSwimlaneCollapsed = hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id);

                html.push(GH.tpl.rapid.swimlane.renderSwimlane({
                    columns: columns,
//                detailViewColumn: detailViewColumnId,
                    swimlane: swimlane,
                    swimlaneStrategy: model.getSwimlaneStrategy(),
                    swimlaneTitle: SwimlaneViewText.getSwimlaneTitle(swimlane, issueCount),
                    cells: cells,
                    issues: issues,
                    index: visibleSwimlaneIndex,
                    collapsed: isSwimlaneCollapsed,
                    issueCount: issueCount,
                    groupData: groupData,
                    displayIssueKeys: displayIssueKeys,
                    selectedIssueKeys: SwimlaneView.getSelectedKeysMap(),
                    mainSelectedIssue: GH.WorkSelectionController.getSelectedIssueKey(),
                    statLabel: SwimlaneView.getStatisticLabel(),
                    isRankable: RankingModel.canUserRank(),
                    showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard()
                }));

                visibleSwimlaneIndex++;
            }
        });

        // show a message if no issues are visible
        if (visibleSwimlaneIndex === 0) {
            html.push(GH.tpl.rapid.swimlane.renderEmptySwimlaneColumns({
                columns: columns,
            }));
        }

        // mark the end of the pool. This would work as well if it was in a DIV surrounding
        // the changing markup, so feel free to change if you don't like marker tags.
        html.push(GH.tpl.rapid.swimlane.renderEndOfPool({rendered: new Date().getTime()}));

        // set the new content
        container[0].innerHTML = html.join('');

        // show a message if no issues are visible
        if (visibleSwimlaneIndex === 0) {
            SwimlaneView.renderActiveSprintEmptyMsg(container);
        }

        // update the banding class, as the number of columns has changed
        //SwimlaneView.updateBandingClass();

        // set constraint busted class on container
        SwimlaneView.setBusted(container);

        // render the release cog
        SwimlaneView.renderReleaseAction();

        container.show();

        // allow event loop to complete and a redraw to happen
        // then register the draggables
        GH.WorkDragAndDrop.initDraggables(container);
    };

    SwimlaneView.renderActiveSprintEmptyMsg = function (container) {
        var hasUser = UserData.hasUser();
        var noActiveSprints = SwimlaneView.rapidViewData.sprintSupportEnabled && !GH.WorkControls.hasSprints();
        var activeQuickFilters = GH.WorkControls.getActiveQuickFilterCount();
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();

        var activeSprintEmptyMsg = new ActiveSprintEmptyView(noActiveSprints, activeQuickFilters, rapidViewId, false, container);

        // check of user can edit the board or not, then show/hide the line 2
        if (hasUser) {
            if (!noActiveSprints && !activeQuickFilters) {
                GH.RapidViewConfig.fetchConfiguration(rapidViewId).done(function (data) {
                    GH.WorkController.trigger(GH.WorkController.EVENT_NAMES.ON_RELOAD_EMPTY_STATE_VIEW, data.canEdit);
                });
            } else {
                // show second line for case of user logged in
                // and the empty state of no active sprint or filtered
                GH.WorkController.trigger(GH.WorkController.EVENT_NAMES.ON_RELOAD_EMPTY_STATE_VIEW, true);
            }
        }

        // save the view into controller, so that it can manage the view of active sprint empty message
        GH.WorkController.setActiveSprintEmptyMsg(activeSprintEmptyMsg);
    };

    /**
     * Re-renders the grid cell in which issueKey is located in
     */
    SwimlaneView.rerenderCellOfIssue = function (issueKey) {
        var position = GridDataController.getModel().getIssuePositionByKey(issueKey);
        if (!position) {
            return;
        }

        SwimlaneView.rerenderCell(position.swimlaneId, position.columnId);
    };

    /**
     * Re-renders a single column
     */
    SwimlaneView.rerenderCell = function (swimlaneId, columnId) {
        var model = GridDataController.getModel();
        var data = {
            swimlane: model.getSwimlaneById(swimlaneId),
            column: model.getColumnById(columnId),
            cells: model.getCells(),
            issues: model.getIssues(),
            groupData: model.getGroupData(),
            displayIssueKeys: SwimlaneView.getCalculatedDisplayIssueKeys(),
            selectedIssueKeys: SwimlaneView.getSelectedKeysMap(),
            mainSelectedIssue: GH.WorkSelectionController.getSelectedIssueKey(),
            statLabel: SwimlaneView.getStatisticLabel(),
            isRankable: RankingModel.canUserRank(),
            showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard()
        };

        // re-render the affected column (inside the affected swimlane)
        var columnHtml = GH.tpl.rapid.swimlane.renderIssueColumn(data);
        var $elem = $(columnHtml);

        // replace the column element
        var container = $('#ghx-pool').find('.ghx-swimlane[swimlane-id="' + swimlaneId + '"]').find('.ghx-columns').find('.ghx-column[data-column-id="' + columnId + '"]');
        container.replaceWith($elem);

        // we need to reregister draggables for the rerendered elements
        // TODO: this could be more specific to only target the actual replaced elements
        GH.WorkDragAndDrop.initDraggables($('#ghx-pool'));
    };

    /**
     * Get the map of selected keys (used for rendering)
     */
    SwimlaneView.getSelectedKeysMap = function () {
        var selectedIssueKeys = GH.WorkSelectionController.getSelectedIssueKeys();
        var keys = {};
        _.each(selectedIssueKeys, function (selectedIssueKey) {
            keys[selectedIssueKey] = true;
        });
        return keys;
    };

    /**
     * Renders the top right column release button
     */
    SwimlaneView.renderReleaseAction = function () {
        // do we have a user, don't show if we don't
        if (!UserData.hasUser()) {
            return;
        }

        // only for kanban mode
        if (SwimlaneView.rapidViewData.sprintSupportEnabled) {
            return;
        }

        // render control
        var lastHeader = $('#ghx-column-headers').find('.ghx-column').last();
        lastHeader.prepend(GH.tpl.rapid.swimlane.renderReleaseVersionControl({
            canRelease: GridDataController.canRelease
        }));
    };

    SwimlaneView.openReleaseVersionDialog = function (e) {
        var $button = $(this);
        if ($button.hasClass('ghx-disabled')) {
            return;
        }

        var rapidViewId = GH.RapidBoard.State.getRapidViewId();
        var columnId = $button.closest('.ghx-column').attr('data-id');
        GH.Dialogs.ReleaseVersionDialog.showDialog(rapidViewId, columnId);
    };


    /**
     * Updates the column quantities class set accordingly to the currently displayed column count.
     */
    SwimlaneView.setBusted = function (targetElement) {
        if (GridDataController.hasBustedColumns()) {
            $(targetElement).addClass('ghx-busted');
        } else {
            $(targetElement).removeClass('ghx-busted');
        }
    };

    /**
     * Get the display issue keys for all issues
     *
     * @param {string} [issueKey] if defined only the key of that issue is calculated
     */
    SwimlaneView.getCalculatedDisplayIssueKeys = function (issueKey) {
        var model = GridDataController.getModel();
        var issueKeyDimensions = GH.WorkView.getIssueKeyDimensions();

        var displayKeys = {};
        if (issueKey) {
            displayKeys[issueKey] = GH.IssueListUtil.buildDisplayIssueKey(issueKey, issueKeyDimensions);
        } else {
            // data model to render
            var issues = model.getIssues();
            _.each(issues, function (issue) {
                displayKeys[issue.key] = GH.IssueListUtil.buildDisplayIssueKey(issue.key, issueKeyDimensions);
            });
        }
        return displayKeys;
    };

    /**
     * Rerenders an issue inside the SwimlaneView.
     *
     * @param issueKey the key of the issue to rerender
     */
    SwimlaneView.rerenderIssue = function (issueKey) {
        var model = GridDataController.getModel();
        var issueData = model.getIssueDataByKey(issueKey);
        if (!issueData) {
            return;
        }

        // update the fake parents
        SwimlaneView.rerenderIssueCard(issueData);

        // update swimlanes
        SwimlaneView.updateSwimlaneHeader(issueData);

        // update fake parents
        SwimlaneView.rerenderFakeParents(issueData);
    };

    /**
     * Updates fake parents
     */
    SwimlaneView.rerenderFakeParents = function (issueData) {
        // stop if this is a child - can't be a fake parent
        if (issueData.parentKey) {
            return;
        }

        // find fake parents
        var fakeParents = $('.js-fake-parent[data-issue-key="' + issueData.key + '"]');
        if (fakeParents.length < 1) {
            return;
        }

        // update title and summary
        var parentStubs = fakeParents.find('.ghx-parent-stub');
        parentStubs.attr('title', issueData.key + ' \u2014 ' + issueData.summary);
        var summaries = parentStubs.find('.ghx-summary');
        summaries.text(issueData.summary);
    };

    /**
     * Rerenders all cards based on a given issue
     * @param issueData
     */
    SwimlaneView.rerenderIssueCard = function (issueData) {
        // update all occurences of that issue
        var issue = $('.js-issue[data-issue-key="' + issueData.key + '"]');
        if (issue.length > 0) {
            // adapt issue key length to the width we got available
            var displayIssueKeys = SwimlaneView.getCalculatedDisplayIssueKeys(issueData.key);

            // render the issue
            var updatedIssue = $(GH.tpl.rapid.swimlane.renderIssue({
                issue: issueData,
                displayIssueKeys: displayIssueKeys,
                selectedIssueKeys: SwimlaneView.getSelectedKeysMap(),
                statLabel: SwimlaneView.getStatisticLabel(),
                showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard()
            }));

            issue.replaceWith(updatedIssue);

            GH.WorkDragAndDrop.registerSingleCardDraggable(updatedIssue);
        }
    };

    /**
     * Updates the swimlane header
     */
    SwimlaneView.updateSwimlaneHeader = function (issueData) {
        // check whether we actually got a swimlane
        var model = GridDataController.getModel();
        var swimlane = model.getParentSwimlaneByIssueKey(issueData.key);
        if (swimlane) {
            // update swimlane
            $(".ghx-swimlane").each(function () {
                var $this = $(this);
                if ($this.attr("swimlane-id") == swimlane.id) {
                    var issueCount = model.getSwimlaneIssueCount(swimlane.id);
                    var swimlaneTitle = SwimlaneViewText.getSwimlaneTitle(swimlane, issueCount);
                    var headerContent = GH.tpl.rapid.swimlane.renderSwimlaneHeaderContent({
                        swimlane: swimlane,
                        swimlaneTitle: swimlaneTitle,
                        swimlaneStrategy: model.getSwimlaneStrategy(),
                        parent: issueData,
                        issueCount: issueCount
                    });
                    var $header = $this.find('.ghx-swimlane-header');
                    $header.html(headerContent);
                    if (issueData.flagged) {
                        $header.addClass("ghx-flagged");
                    } else {
                        $header.removeClass("ghx-flagged");
                    }
                }
            });

            // update stalker
            GH.SwimlaneStalker.updateSwimlaneHeaderStalker();
//        GH.SwimlaneStalker.updateStalkerContent(swimlane.id);
        }
    };

    /**
     * Called whenever the size changes.
     */
    SwimlaneView.handleResizeEvent = function () {
        // no-op if we are not displayed
        if (!SwimlaneView.visible) {
            return;
        }

        //SwimlaneView.updateBandingClass();
    };

    /**
     * Responsive Issues Classes - used to change stuff depending on available space
     */

    /**
     * Collapse/Expand 1 Swimlane
     */
    SwimlaneView.toggleCollapsedHandler = function (event) {
        if (event.type === "keydown") {
            // was a key press that wasn't Enter or Space
            if (event.keyCode !== 13 && event.keyCode !== 32) {
                return;
            }
            event.preventDefault(); // stop space from scrolling
        }
        var swimlane = $(this).closest('.ghx-swimlane');
        var collapsed = !swimlane.hasClass('ghx-closed');

        var $expander = $(event.target).closest(".js-expander");
        $expander.attr('aria-expanded', !collapsed);

        SwimlaneView.setCollapsed(swimlane, collapsed);
    };

    SwimlaneView.setCollapsed = function (swimlane, collapsed) {
        var swimlaneId = parseInt(swimlane.attr('swimlane-id'), 10);
        // toggle and store change
        if (collapsed) {
            swimlane.addClass('ghx-closed');
        } else {
            swimlane.removeClass('ghx-closed');
        }

        // store the setting
        SwimlaneView.setCollapsedState(swimlaneId, collapsed);

        // collapsing/expanding changes the dom, ensure we properly relayout the detail view
        Layout.fireDelayedWindowResize();
    };

    SwimlaneView.setCollapsedById = function (swimlaneId, collapsed) {
        $(".ghx-swimlane").each(function () {
            var $this = $(this);
            if ($this.attr("swimlane-id") == swimlaneId) {
                if ($this.hasClass('ghx-closed') != collapsed) {
                    SwimlaneView.setCollapsed($this, collapsed);
                }
            }
        });
    };

    /**
     * Collapse All Swimlanes
     * @param e (optional)
     */
    SwimlaneView.collapseAll = function (e) {
        // don't follow through if disabled
        if (e && $(e.target).closest('li').hasClass('disabled')) {
            return false;
        }

        $('.ghx-swimlane')
            .addClass('ghx-closed')
            .each(function () {
                SwimlaneView.setCollapsedState(parseInt($(this).attr('swimlane-id')), true);
            });
    };

    /**
     * Expand All Swimlanes
     * @param e (optional)
     */
    SwimlaneView.expandAll = function (e) {
        // don't follow through if disabled
        if (e && $(e.target).closest('li').hasClass('disabled')) {
            return false;
        }

        $('.ghx-swimlane')
            .removeClass('ghx-closed')
            .each(function () {
                SwimlaneView.setCollapsedState(parseInt($(this).attr('swimlane-id')), false);
            });
    };

    /**
     * Toggle the expanded/collapsed state of all swimlanes (triggered by keyboard shortcut).
     * The algorithm is:
     *  - if all swimlanes are currently collapsed, then expand all
     *  - otherwise, collapse all
     */
    SwimlaneView.toggleAllSwimlanes = function () {
        var model = GridDataController.getModel();

        // do we even have swimlanes?
        if (!model.hasSwimlanes()) {
            return false;
        }

        var swimlanes = model.getSwimlanes();
        var allCollapsed = true;
        _.any(swimlanes, function (swimlane) {
            if (!GH.RapidBoard.State.isSwimlaneCollapsed(swimlane.id)) {
                // Always returns false for empty swimlanes although they are not visible so check for this
                // When sorted by Assignee no swimlanes are empty but they do not have issueIds
                var hasIssues = (swimlane.issueIds) ? swimlane.issueIds.length > 0 : true;
                if (hasIssues) {
                    allCollapsed = false;
                    return true;
                }
            }
        });

        if (allCollapsed) {
            SwimlaneView.expandAll();
        } else {
            SwimlaneView.collapseAll();
        }
    };

    /**
     * Checks whether a swimlane is expanded, expands otherwise and returns true if done so
     * @param issueKey
     */
    SwimlaneView.ensureSwimlaneExpandedForIssueKey = function (issueKey) {
        var model = GridDataController.getModel();
        var swimlaneId = model.getSwimlaneIdByIssueKey(issueKey);
        var hasSwimlanes = model.hasSwimlanes();
        var isCollapsed = hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlaneId);
        if (!isCollapsed) {
            return false;
        }

        // expand
        SwimlaneView.setCollapsedById(swimlaneId, false);
        return true;
    };

    /**
     * Set the swimlane collapsed for a given issue key
     * Note: This method does NOT update the ui
     */
    SwimlaneView.setSwimlaneCollapsedStateForIssueKey = function (issueKey, collapsed) {
        var model = GridDataController.getModel();
        var swimlaneId = model.getSwimlaneIdByIssueKey(issueKey);
        var hasSwimlanes = model.hasSwimlanes();
        var isCollapsed = hasSwimlanes && GH.RapidBoard.State.isSwimlaneCollapsed(swimlaneId);
        if (isCollapsed != collapsed) {
            SwimlaneView.setCollapsedState(swimlaneId, collapsed);
        }
    };

    /**
     * Sets the collapsed state for a swimlane
     */
    SwimlaneView.setCollapsedState = function (swimlaneId, collapsed) {
        GH.RapidBoard.State.setSwimlaneCollapsed(swimlaneId, collapsed);
    };

    SwimlaneView.getStatisticLabel = function () {
        return SwimlaneView.rapidViewConfig.trackingStatistic
        && SwimlaneView.rapidViewConfig.trackingStatistic.isEnabled ?
            SwimlaneView.rapidViewConfig.trackingStatistic.name :
            SwimlaneView.rapidViewConfig.estimationStatistic.name;
    };

    return SwimlaneView;
});

AJS.namespace('GH.SwimlaneView', null, require('jira-agile/rapid/ui/work/swimlane-view'));