define('jira-agile/rapid/ui/work/work-controller', [
    "jira-agile/rapid/ui/work/swimlane-view",
    "jira-agile/rapid/ui/work/ranking-model",
    "jira-agile/rapid/ui/work/grid-data-controller",
    "jira-agile/gh-features",
    "jira-agile/rapid/logger",
    "jira-agile/rapid/ajax",
    "jira-agile/rapid/ui/notification",
    "jira-agile/rapid/ui/plan/sprint-config",
    "jquery",
    "jira/jquery/deferred",
    'jira-agile/rapid/ui/kanplan/KanPlanOnboardingController',
    'underscore',
    'jira/analytics',
    'jira-agile/rapid/global-events',
    'jira-agile/rapid/analytics-tracker',
    'jira/util/formatter',
    'jira/featureflags/feature-manager',
    'require',
    'jira-agile/rapid/ui/work/work-data-loader',
    'jira-agile/rapid/ui/work/ActiveSprintEmptyView',
    'jira-agile/rapid/ui/kanplan/kan-plan-feature-service'
], function(
    SwimlaneView,
    RankingModel,
    GridDataController,
    Features,
    Logger,
    Ajax,
    Notification,
    SprintConfig,
    $,
    Deferred,
    KanPlanOnboardingController,
    _,
    jiraAnalytics,
    GlobalEvents,
    AnalyticsTracker,
    formatter,
    FeatureManager,
    require,
    WorkDataLoader,
    ActiveSprintEmptyView,
    KanPlanFeatureService
) {
    "use strict";

    /**
     * Work Controller
     */
    let WorkTransitionService;
    let WorkDragAndDrop;
    let EditableDetailsViewReloadReason;

    GlobalEvents.on('pre-initialization', () => {
        WorkTransitionService = require('jira-agile/rapid/ui/work/work-transition-service');
        WorkDragAndDrop = require('jira-agile/rapid/ui/work/work-drag-and-drop');
    });

    const WorkController = new GH.Events();

    WorkController.analytics = new AnalyticsTracker("gh.workmode");

    WorkController.viewData = undefined;

    WorkController.currentViewConfig = {};

    WorkController.isVisible = false;

    WorkController.CALLBACK_POOL_RENDERED = 'gh.work.pool.rendered';
    WorkController.CALLBACK_POOL_LOADED = 'gh.work.pool.loaded';
    WorkController.EVENT_NAMES = {
        ON_PLAN_MODE: 'OnPlanMode',
        ON_ALL_FILTERS_CLEARING: 'OnAllFiltersClearing',
        ON_RELOAD_EMPTY_STATE_VIEW: 'OnReloadEmptyStateMessage'
    };

    WorkController.TRANSITION = "transition";
    WorkController.QUICK_EDIT = "quickEdit";

    /**
     * Initializes the view controller.
     */
    WorkController.init = _.once(function () {
        // init controls (quick filters, etc)
        GH.WorkControls.init();

        // keyboard shortcuts
        GH.WorkKBNavigation.init();

        // Work View
        GH.WorkView.init();

        // SwimlaneView
        SwimlaneView.init();

        // issue
        GH.WorkSelectionController.init();

        // drag and drop
        GH.WorkDragAndDrop.init();

        // handle issue selection events
        $(GH).bind('issueSelected', WorkController.handleIssueSelectionEvent);

        // handle quick filter changes
        $(GH.WorkControls.quickfilters).bind('quickFilterChange', WorkController.reload);
        $(GH.WorkControls.sprintfilters).bind('sprintFilterChange', WorkController.reload);

        // handle new issue created
        $(GH).bind('issueCreated', WorkController.handleIssueCreatedEvent);

        // listen to updates to an issue.
        $(GH).bind('issueUpdated', WorkController.handleIssueUpdatedEvent);

        // listen to issues removed from sprint event
        $(GH).bind('issuesRemovedFromSprint', WorkController.handleIssuesRemovedFromSprint);

        // handle connection timeout while releasing
        $(GH).bind(GH.Dialogs.ReleaseVersionDialog.EVENT_TIMEOUT, WorkController.reload);

        KanPlanOnboardingController.init();
    });

    /**
     * Sets the rapid view data.
     */
    WorkController.setRapidView = function (rapidViewData) {
        WorkController.rapidViewData = rapidViewData;
        SwimlaneView.setRapidView(WorkController.rapidViewData);
    };

    WorkController.isSprintSupportEnabled = function () {
        return WorkController.rapidViewData.sprintSupportEnabled;
    };

    /**
     * Called when the work tab is shown
     * @return {promise}
     */
    WorkController.show = function () {
        Logger.log("WorkController.show", Logger.Contexts.ui);
        if (!WorkController.rapidViewData) {
            return;
        }
        WorkController.isVisible = true;

        // initialize the view
        GH.WorkView.show();

        // set the correct rapid view id
        GH.DetailsView.setRapidViewId(WorkController.rapidViewData.id);

        // kick off loading the view configuration
        return WorkController.loadRapidViewConfig();
    };

    /**
     * Called when the work tab is hidden
     */
    WorkController.hide = function () {
        Logger.log("WorkController.hide", Logger.Contexts.ui);

        // stop the poller
        GH.Poller.removePoller('WorkPoller');

        // hide the swimlane view
        SwimlaneView.hide();

        // hide the whole work view
        GH.WorkView.hide();
        GH.DetailsView.hide();

        WorkController.isVisible = false;

        // mark the current data as dirty
        WorkDataLoader.markDirty();

        KanPlanOnboardingController.hide();
    };

    /**
     * Is the work tab currently active, thus is the pool displayed?
     */
    WorkController.isActive = function () {
        return WorkController.isVisible;
    };

    /**
     * Load the rapid view configuration used by the pool
     */
    WorkController.loadRapidViewConfig = function () {
        // create the promise for initializing view
        var deferred = new Deferred();

        // callback called when the view configuration has been loaded
        var callback = function (data) {
            // stop here if the pool isn't displayed anymore
            if (!WorkController.isVisible) {
                return;
            }

            WorkController.currentViewConfig = data;

            var promise = WorkController.initializeWorkMode();

            // invoke done callback
            promise.done(function (data) {
                deferred.resolveWith(this);
                WorkController._sendOnLoadAnalytics(data);
            });

            promise.fail(function () {
                deferred.rejectWith(this);
            });
        };

        // fetch the configuration
        GH.RapidViewConfig.fetchConfiguration(WorkController.rapidViewData.id).done(callback);

        return deferred.promise();
    };

    WorkController._sendOnLoadAnalytics = function (data) {
        const DEFAULT_EPICS_COUNT = -1;
        var dataModel = GridDataController.getModel();

        if (!dataModel) {
            return;
        }

        var properties = {
            boardId: GH.RapidBoard.State.getRapidViewId(),
            issuesCount: _.size(dataModel.getIssues()),
        };

        if (GH.RapidBoard.State.isKanbanBoard()) {
            properties.isBacklogEnabled = KanPlanFeatureService.shouldShowKanbanBacklog();
            properties.epicsPanelEnabled = KanPlanFeatureService.shouldShowEpicsPanel();
            if (dataModel.data && dataModel.data.columns && dataModel.data.columns.length > 0 && dataModel.data.columns[0].stats) {
                properties.firstColumnIssueCount = dataModel.data.columns[0].stats.total;
            }
        }

        properties.epicsCount = DEFAULT_EPICS_COUNT;
        if (data && data.epicData) {
            properties.epicsCount = data.epicData.epicsCount || DEFAULT_EPICS_COUNT;
        }

        jiraAnalytics.send({
            name: 'jira-software.' + GH.RapidBoard.State.getBoardType() + '.work.view',
            properties: properties
        });
    };

    /**
     * Initialize the pool
     * @return {promise}
     */
    WorkController.initializeWorkMode = function () {
        // we can only draw something if we got columns in the view
        if (!WorkController.currentViewConfig.columns || WorkController.currentViewConfig.columns.length < 0) {
            // TODO: empty everything
            return;
        }

        // if the new view config has the same id as the old one, then don't redraw the quick filters
        GH.WorkControls.setRapidViewConfig(WorkController.currentViewConfig);
        SwimlaneView.setRapidViewConfig(WorkController.currentViewConfig);
        GH.DetailsView.setRapidViewConfig(WorkController.currentViewConfig);

        // render work controls
        GH.WorkControls.renderControls();

        // hide the loading transition for QF
        GH.WorkView.hideLoadingFilter();

        // make sure we appropriately size the view
        GH.RapidBoard.ViewController.updateContentContainer();

        // and load the pool data
        return WorkController.loadPoolData();
    };

    WorkController.optimisticallyTransition = function (issueKeys, selectedTransitionId, transitionStatus, sourceColumn, targetColumn) {
        const rapidViewId = GH.RapidBoard.State.getRapidViewId();
        const request = {
            rapidViewId,
            issueKeys,
            targetColumn,
            selectedTransitionId
        };

        const transitionIssueOnServerPromise = WorkController.submitTransitionAndRank(request);

        // Move issues in the UI. This may take non-trivial time on a large board.
        // This method computes synchronously, which is why we do it after submitting the request to the server.
        WorkController.moveIssuesTo(issueKeys, sourceColumn, targetColumn);

        transitionIssueOnServerPromise.done(() => {
            issueKeys.forEach((issueKey) => {
                WorkController.handleIssueUpdatedEvent({}, {
                    issueId: GridDataController.getModel().getIssueIdForKey(issueKey),
                    source: WorkController.TRANSITION
                });
            });

            // Show the resolve parent dialog if required. (We can pick any issue
            // as you can only select siblings of the same parent)
            GH.Dialog.createParentCompletedCheckCallback(issueKeys[0])();
        }).fail(handleOffline).fail(() => {
            // We modified some stuff, clear the etag to force a full reload from server.
            WorkDataLoader.clearEtag();
            WorkController.reload();
        });
    };
    /**
     * Moves the issues in the UI. Does not contact the server. Completes synchronously.
     */
    WorkController.moveIssuesTo = function (issueKeys, sourceColumn, targetColumn) {
        GridDataController.getModel().moveIssuesToColumn(issueKeys, sourceColumn, targetColumn);

        // The details view will update when the handleIssueUpdatedEvent is invoked.
        // So we'll skip it this time.
        GH.WorkView.skipNextDetailViewUpdate();

        WorkController.updateUI();
    };

    WorkController.submitTransitionAndRank = function (request) {
        return Ajax.put({
            url: '/xboard/transitionAndRank',
            data: request
        });
    };

    function handleOffline(xhr) {
        if (xhr && xhr.readyState !== 4) {
            Notification.showError("Error", formatter.I18n.getText('gh.rapid.board.transition.offline'));
        }
    }


    WorkController.preload = function (viewId) {
        // load the data, but ignore the result
        WorkDataLoader.getData(viewId);
    };

    /**
     * Reload the pool data entirely
     * @return {promise}
     */
    WorkController.reload = function () {
        if (!WorkController.isVisible) {
            var viewId = WorkController.rapidViewData.id;
            var promise = WorkDataLoader.getData(viewId);
            return promise;
        }

        // Mark the current data as dirty
        WorkDataLoader.markDirty();

        // clear updated messages
        Notification.clearBoardUpdatedMessage();

        // then load new data
        return WorkController.loadPoolData();
    };

    /**
     * Load the pool data
     * @return {promise}
     */
    WorkController.loadPoolData = function () {

        // show the loading transition for QF
        GH.WorkView.showLoadingPool();

        // execute the request
        var viewId = WorkController.rapidViewData.id;
        var promise = WorkDataLoader.getData(viewId);
        promise.done(function (data) {
            WorkController.setPoolData(data);
        });

        // handle global errors (they'll all be global in this case)
        promise.fail(WorkController.handleInvalidSprint);

        // cleanup the loading spinner
        promise.fail(function () {
            GH.WorkView.hideLoadingPool();
        });

        return promise;
    };

    WorkController.handleInvalidSprint = function (res, onGlobalError, globalErrorDismissable) {
        let err = res.error && res.error.errors && res.error.errors[0] && res.error.errors[0].message;

        if (err && err === formatter.I18n.getText('gh.sprint.error.not.found')) {
            GH.WorkControls.sprintfilters.validateActiveSprints();
        } else if (err && err === formatter.I18n.getText('gh.sprint.error.no.mapped.columns')) {
            WorkController.renderNoMappedColumnsMsg();
        } else {
            Ajax.handleGlobalErrors(res, onGlobalError, globalErrorDismissable);
        }
    };

    WorkController.checkForUpdates = function (updateImmediately) {
        var viewId = WorkController.rapidViewData.id;
        return WorkDataLoader.checkForUpdates(viewId).done(function (result) {
            Logger.log('WorkController update check: ' + result);
            if (result === WorkDataLoader.NEW_DATA_WITH_CHANGES) {
                if (updateImmediately) {
                    WorkController.applyLoadedData();
                } else {
                    WorkController.showBoardUpdatedMessage();
                }

                return;
            }

            if (updateImmediately && Notification.isBoardUpdatedMessageVisible()) {
                WorkController.applyLoadedData();
                return;
            }

            if (result === WorkDataLoader.NEW_DATA_NO_CHANGE) {
                // we received changes, but they don't affect the board.
                // ask the detail view to check for updates

                // TODO: the detail view doesn't have a separate update checker code, so for now all we can do is ask it to
                //       reload. For now we'll only do this in case we update everything immediately anyways,
                //       as we otherwise could reload the detail view while a user is editing the issue.
                if (updateImmediately) {
                    WorkController.reloadDetailView();
                } else {
                    // to be done
                }
            }
        });
    };

    WorkController.showBoardUpdatedMessage = function () {
        Notification.showBoardUpdatedMessage().done(function () {
            // apply the loaded data
            WorkController.applyLoadedData();
        });
    };

    /**
     * Updates already loaded data
     */
    WorkController.applyLoadedData = function () {
        // execute the request (which will directly return as the data has already been loaded)
        var viewId = WorkController.rapidViewData.id;
        var promise = WorkDataLoader.getData(viewId);
        promise.done(function (data) {
            // update
            WorkController.setPoolData(data);

            // then clear the message
            Notification.clearBoardUpdatedMessage();
        });
    };

    /**
     * Checks whether the newly provided pool data differs from what is currently used.
     */
    WorkController.hasDataChanged = function (data) {
        return GridDataController.hasDataChanged(data);
    };

    /**
     * Set the new pool data, rerender the view
     * @fires WorkController#after:render when the board has finished rendering and the user is unblocked.
     */
    WorkController.setPoolData = function (data) {
        if (!WorkController.isVisible) {
            return;
        }

        if (FeatureManager.isFeatureEnabled(WorkDragAndDrop.OPTIMISTIC_TRANSITIONS_FLAG) && !WorkTransitionService.tooManyProjects) {
            WorkTransitionService.preloadData(data.issuesData.projects);
        }

        // set the data in the grid controller
        GridDataController.setData(data);
        GH.CallbackManager.executeCallbacks(WorkController.CALLBACK_POOL_LOADED);

        WorkController.setSprintData(data);
        // set the sprints
        if (data.sprintsData) {
            GH.WorkControls.setSprintData(data.sprintsData.sprints, data.sprintsData.canManageSprints);
        } else {
            GH.WorkControls.setSprintData([], false);
        }

        // initialize issue selection related stuff
        WorkController.initializeSelection();

        // ensure that the swimlane of the selected issue column is expanded
        WorkController.ensureSelectionExpanded();

        // update the ui
        WorkController.updateUI();

        GH.CallbackManager.executeCallbacks(WorkController.CALLBACK_POOL_RENDERED);

        WorkController.trigger('after:render');

        // register or reschedule the poller
        GH.Poller.addPoller('WorkPoller', WorkController.checkForUpdates);
    };

    WorkController.setSprintData = function (data) {
        WorkController.sprintData = data.sprintsData;
    };

    /**
     * Initializes issue selection related functionality
     */
    WorkController.initializeSelection = function () {
        // set the magic index
        GH.WorkKBNavigation.initCurrentPosition();

        // update the selection state in the data controller
        GH.WorkSelectionController.validateCurrentSelection();
    };

    /**
     * Expands the swimlane of the selected issue if necessary
     */
    WorkController.ensureSelectionExpanded = function () {
        // fetch selection
        var selectedIssueKey = GH.WorkSelectionController.getSelectedIssueKey();
        if (!selectedIssueKey) {
            return;
        }

        // valid selection, ensure the swimlane is not collapsed
        SwimlaneView.setSwimlaneCollapsedStateForIssueKey(selectedIssueKey, false);
    };

    /**
     * Update the UI
     */
    WorkController.updateUI = function () {
        // prepare the view (detail view, selected columns)
        WorkController.configureUI();

        // actually render the ui
        WorkController.renderUI();

        // scroll the issue to view
        GH.WorkView.scrollIssueToView();

        // fire an event (for when our container wants to do something now - eg: gadgets need to resize)
        $(GH).trigger('workModeUIReady');
    };

    /**
     * Configure the components for rendering
     */
    WorkController.configureUI = function () {
        // fetch the detail view state, ensure the detail view is closed if no issue is selected
        var selectedIssueKey = GH.WorkSelectionController.getSelectedIssueKey();
        var detailViewOpened = WorkController.isDetailsViewOpened();
        if (!selectedIssueKey && detailViewOpened) {
            // mark detail view as closed
            WorkController.setDetailViewOpenedState(false);
            detailViewOpened = false;
        }

        // configure the work view
        GH.WorkView.configureUI(detailViewOpened ? selectedIssueKey : false);
    };

    /**
     * Render the view according to the current configuration
     */
    WorkController.renderUI = function () {
        // show sprints
        GH.WorkControls.drawSprints();

        // render the pool
        GH.WorkView.renderPoolAndDetailView();

        // scroll the issue to view
        GH.WorkView.scrollIssueToView();

        var gridDataModel = GridDataController.getModel();

        KanPlanOnboardingController.update(gridDataModel);

        // update swimlane options in the View Actions
        GH.WorkControls.updateSwimlaneOptions(gridDataModel.hasSwimlanes());

        // make sure the content container is properly layed out
        GH.RapidBoard.ViewController.updateContentContainer();
    };


// Detail view

    /**
     * Set the new detail view opened state
     */
    WorkController.setDetailViewOpenedState = function (opened) {
        GH.RapidBoard.State.setWorkViewDetailsViewOpened(opened || false);
    };

    /**
     * Is the detail view opened
     */
    WorkController.isDetailsViewOpened = function () {
        return GH.RapidBoard.State.isDetailsViewOpened();
    };

    WorkController.getStatisticField = function () {
        return WorkController.currentViewConfig.statisticConfig.typeId;
    };
    /**
     * Toggles the details view
     */
    WorkController.toggleDetailsView = function () {
        if (WorkController.isDetailsViewOpened()) {
            WorkController.closeDetailsView();
        } else {
            var selectedIssueKey = GH.WorkSelectionController.getSelectedIssueKey();
            WorkController.openDetailsView(selectedIssueKey);
        }
    };

    /**
     * Opens the details view, displaying the issue specified.
     * Does nothing if no issue is specified.
     */
    WorkController.openDetailsView = function openDetailsView(issueKey) {
        // if we don't have a key then there's nothing to do
        if (!issueKey) {
            return;
        }

        // ensure that the issue specified is selected
        var selectedIssueKey = GH.WorkSelectionController.getSelectedIssueKey();
        var stateChanged = false;
        if (!selectedIssueKey || selectedIssueKey != issueKey) {
            // signal a state change
            stateChanged = true;
            GH.WorkSelectionController.selectIssue(issueKey, {pushState: false});
        }

        // if details view is already visible, simply trigger loading the new details
        if (WorkController.isDetailsViewOpened()) {
            WorkController.updateDetailViewIssue(issueKey);

            // otherwise open the details view
        } else {
            // signal a state change
            stateChanged = true;
            WorkController.setDetailViewOpenedState(true);

            // update the view
            GH.WorkView.openOrCloseDetailView(issueKey);
        }

        // only push state if we changed our state somehow
        if (stateChanged) {
            GH.RapidBoard.State.pushState();
        }
    };

    /**
     * Closes the detail view.
     */
    WorkController.closeDetailsView = function closeDetailsView() {
        // ignore if there are errors in edit fields
        if (GH.DetailsView.hasCurrentError()) {
            return;
        }
        // ignore if already closed
        if (!WorkController.isDetailsViewOpened()) {
            return;
        }

        // mark the details view as closed and redraw
        WorkController.setDetailViewOpenedState(false);
        GH.RapidBoard.State.pushState();

        // update the view
        GH.WorkView.openOrCloseDetailView(false);
    };

    /**
     * Reload the detail view, e.g. after the selected issue changed
     */
    WorkController.reloadDetailView = function (detailsViewLoadReason) {
        // ignore if not visible
        if (!WorkController.isDetailsViewOpened()) {
            return;
        }

        GH.DetailsView.reload(detailsViewLoadReason);
    };

    /**
     * Sets the new details view issue.
     */
    WorkController.updateDetailViewIssue = function (issueKey) {
        if (!WorkController.isDetailsViewOpened()) {
            return;
        }

        // set the selected issue key
        GH.DetailsView.setSelectedIssueKey(issueKey);

        // simply trigger show again
        GH.DetailsView.show();
    };


    /**
     * Toggles between showing and hiding the epic lozenges.
     */
    WorkController.toggleEpicsShowOnWorkboard = function () {
        GH.ViewActions.toggleEpicsShowOnRapidBoard();
        WorkController.updateUI();
    };

    /**
     * Handles issue selection events
     */
    WorkController.handleIssueSelectionEvent = function (event, issueKey) {
        // fetch the newly selected issue
        var selectedIssueKey = GH.WorkSelectionController.getSelectedIssueKey();

        // if the detail view is open we need to tell it to update as well as ensure the right pool column is shown
        if (WorkController.isDetailsViewOpened()) {
            if (selectedIssueKey) {
                // set the new issue
                WorkController.updateDetailViewIssue(selectedIssueKey);
            } else {
                // close it, because there is nothing to display
                WorkController.closeDetailsView();
            }
        }
    };

    WorkController.resolveParentTask = function (issue) {
        if (!WorkController.isVisible) {
            return;
        }

        Ajax.get({
            url: '/xboard/issue/subtasksInFinalColumn.json',
            data: {
                "issueId": issue.id,
                "rapidViewId": WorkController.rapidViewData.id
            }
        }, 'subtaskCompletionCheck')
            .done(function (data) {
                WorkController.loadTransitions(issue, data);
            })
            .fail(GH.WorkView.hideLoadingPool);
    };

    WorkController.loadTransitions = function (issue, data) {
        // stop here if the pool isn't displayed anymore
        if (!WorkController.isVisible) {
            return;
        }

        // load transitions if all subtasks are in the final column
        if (data.subtasksInFinalColumn) {
            // TODO Should we get a fresh parent issue from the server? Or is this good enough?
            Ajax.get({
                url: '/xboard/issue/transitions.json',
                data: {
                    'issueId': issue.id
                }
            }, "loadTransitions")
                .done(function (data) {
                    // TODO Merge this and the similar code in DragAndDrop
                    var model = GridDataController.getModel();

                    var columnData = _.last(model.data.columns);
                    if (!columnData) {
                        return;
                    }

                    // for each column, find the transitions that match
                    var matchingTransitions = [];
                    for (var x = 0; x < data.transitions.length; x++) {
                        for (var i = 0; i < columnData.statusIds.length; i++) {
                            if (columnData.statusIds[i] == data.transitions[x].targetStatus) {
                                matchingTransitions.push(data.transitions[x]);
                            }
                        }
                    }
                    GH.Dialogs.CompleteParentDialog.showDialog(issue, matchingTransitions, columnData);
                });
        }
    };

// Events

    /**
     * Handle issue updated events
     */
    WorkController.handleIssueUpdatedEvent = function (event, data) {
        if (!WorkController.isVisible) {
            return;
        }

        let reloadAllData = false;
        let needsVersionReload = false;

        Logger.log(
            event.type + " from source " + data.source + " handled",
            Logger.Contexts.ui
        );

        // reload if the sprint field has changed
        if (data.fieldChanges) {
            var sprintFieldChanges = data.fieldChanges[SprintConfig.getSprintFieldId()];

            if (sprintFieldChanges && sprintFieldChanges.original != sprintFieldChanges.updated) {
                reloadAllData = true;
            }
        }
        if (data.source === WorkController.TRANSITION || data.source === WorkController.QUICK_EDIT) {
            WorkDataLoader.markDirty();
            needsVersionReload = true;
        }
        if (reloadAllData) {
            WorkController.reload();
        } else {
            // load issue again
            WorkController.loadIssue(data.issueId);

            // update detail view in case the event came from quickEdit (otherwise it came from detail view, which is already
            // up to date
            if (needsVersionReload) {
                if (Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled()) {
                    var editedIssueModel = GridDataController.getModel().getIssueDataById(data.issueId);
                    if (editedIssueModel && editedIssueModel.parentId) {
                        EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
                        WorkController.reloadDetailView(EditableDetailsViewReloadReason.SUBTASKS_CHANGED);
                    } else {
                        WorkController.reloadDetailView();
                    }
                } else {
                    WorkController.reloadDetailView();
                }
            }
            if (data.source === WorkController.TRANSITION) {
                if(!EditableDetailsViewReloadReason) {
                    EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
                }
                WorkController.reloadDetailView(EditableDetailsViewReloadReason.DETAILS_FIELD_CHANGED);
            }
        }
    };

    /**
     * Loads the data for an issue, and updates the view
     */
    WorkController.loadIssue = function (issueId) {
        Ajax.get({
            url: '/xboard/work/issue.json',
            data: {
                'rapidViewId': GH.RapidBoard.State.getRapidViewId(),
                'issueId': issueId
            }
        }, 'loadIssue.' + issueId)
            .done(function (issue) {
                // update the data model
                var model = GridDataController.getModel();
                if (!model) {
                    // could happen in case an inline edit is completed after switching from plan to work
                    return;
                }
                model.updateIssueData(issue);

                // Only re-render the issue if it is not being dragged (otherwise it will fly out of the user's hand).
                // It will re-render again when it is dropped.
                if (!GH.WorkDragAndDrop.isIssueBeingDragged(issue.key)) {
                    // re-render that issue
                    SwimlaneView.rerenderIssue(issue.key);
                }
            });
    };


    /**
     * Handle issue created events
     */
    WorkController.handleIssueCreatedEvent = function (event, data) {
        if (!WorkController.isVisible) {
            return;
        }

        // Ajax requests on Firefox fail if issued through an (escape) key event, IF the key event propagation is not stopped.
        // This event could come from quick create (create an issue with "create another" selected, then hit escape)
        // Workaround: do the actual action after a timeout, which will happen outside the keyboard event.
        setTimeout(function () {
            WorkController.reloadAndSelectCreatedIssues(data);
        }, 0);
    };

    /**
     * Reloads the board, then selects the created issue
     */
    WorkController.reloadAndSelectCreatedIssues = function (data) {
        // fetch the issues
        var issues = data.issues;
        var issueKeys = _.pluck(issues, 'issueKey');

        // reload the work board
        var reloadPromise = WorkController.reload();

        // check issues can appear on board
        var matchPromise = Ajax.get({
            url: '/xboard/issue/matchesBoard',
            data: {
                rapidViewId: WorkController.rapidViewData.id,
                issueIdsOrKeys: issueKeys
            }
        }).andThen(function (result) {
            // did all issues match?
            return {
                issuesMatchingBoard: result.issueIdsOrKeys,
                issuesAddedToBacklog: result.issuesInBacklog,
                allIssuesMatchBoard: result.issueIdsOrKeys && (result.issueIdsOrKeys.length === issueKeys.length),
                allIssuesAddedToBacklog: result.issuesInBacklog && (result.issuesInBacklog.length === issueKeys.length)
            }
        }, function () {
            // on failure (e.g. error response due to insufficient permissions), just resolve with 'false' (no match)
            return Deferred().resolve(false);
        });

        return $.when(reloadPromise, matchPromise).done(function (reloadResult, matchResult) {
            WorkController._issueCreatedCallback(issueKeys, data, matchResult);
        });
    };

    /**
     * Creates a callback that selects the created issues or displays a message in case creation failed
     */
    WorkController._issueCreatedCallback = function (issueKeys, data, matchResult) {
        // check whether some or all issues are valid (at this point we got the reloaded model)
        var model = GridDataController.getModel();

        var validIssueKeys = _.filter(issueKeys, model.isIssueValid, model);

        // select issue - if it's not an epic
        var lastIssueKey = false;
        if (validIssueKeys.length) {
            lastIssueKey = _.last(validIssueKeys);
            GH.WorkSelectionController.selectIssue(lastIssueKey); // select last issue we created
        }

        // could the issue be selected?
        var issueSelected = lastIssueKey && GH.WorkSelectionController.isInSelection(lastIssueKey);
        // if issue can't be selected or if other issues are added to backlog, display message
        if (!issueSelected || matchResult.issuesAddedToBacklog.length) {
            var workModeData = _.extend({}, WorkController.sprintData, matchResult);
            GH.RapidBoard.QuickCreate.showCreatedIssuesMessage(data, workModeData);
            $(".js-add-new-issue-to-sprint").unbind("click");
            $(".js-add-new-issue-to-sprint").click(WorkController.addNewIssueToSprint);
        }
    };

    WorkController.addNewIssueToSprint = function (e) {
        Notification.clear();
        WorkController.analytics.trigger("addtosprintaftercreate");
        var link = $(e.target);
        var sprintId = link.attr("data-sprint-id");
        var issueKeys = link.attr("data-issue-keys").split(",");
        var model = GridDataController.getModel();
        var rankAfterKey = _.last(model.data.order);

        GH.RankController.sprintRankIssues(RankingModel.getRankCustomFieldId(), issueKeys, sprintId, null, rankAfterKey)
            .done(function (response) {

                // although the response could contain partial success (i.e. some issues were not ranked properly), we ignore
                // it as on Work Mode, adding new issues to the current sprint does not have a visible difference if they
                // are not explicitly ranked beneath the last issue in the sprint.
                var errorRankResult = GH.RankController.getErrorRankResultFromResponse(response);
                if (errorRankResult) {
                    Logger.log(
                        "Rank operation after adding new issues to sprint did not completely succeed.",
                        Logger.Contexts.callback
                    );
                }

                var promise = WorkController.reload();
                promise.done(function () {
                    model = GridDataController.getModel();

                    var issueToSelect = false;
                    var issueFiltered = false;
                    _.each(issueKeys, function (issueKey) {
                        if (model.isIssueValid(issueKey)) {
                            issueToSelect = issueKey;
                        } else {
                            issueFiltered = true;
                        }
                    });

                    if (issueToSelect) {
                        GH.WorkSelectionController.selectIssue(issueToSelect);
                    }

                    if (issueFiltered) {
                        // show a message because none of the issues created were able to be shown
                        Notification.showSuccess(formatter.I18n.getText('gh.rapid.create.issue.add.to.sprint.filtered', issueKeys.length), {timeout: 10000});
                    }
                });
            });

    };

// just reload
    WorkController.handleIssuesRemovedFromSprint = WorkController.reload;

    WorkController.changeModeToPlan = function () {
        GH.RapidBoard.ViewController.setMode('plan');
    };
    WorkController.on(WorkController.EVENT_NAMES.ON_PLAN_MODE, WorkController.changeModeToPlan);

    WorkController.clearAllFiltersAndReload = function () {
        var quickFilterControl = GH.WorkControls.quickfilters;
        if (quickFilterControl) {
            quickFilterControl.clearFilters();
            GH.RapidBoard.State.pushState();
        }

        // reload the board
        WorkController.reload();
    };
    WorkController.on(WorkController.EVENT_NAMES.ON_ALL_FILTERS_CLEARING, WorkController.clearAllFiltersAndReload);

    /**
     * Show the no mapped columns message
     */
    WorkController.renderNoMappedColumnsMsg = function () {
        // check if the message is exist or not before show
        var emptyStateMsgView = $('.ghx-no-active-sprint-message');
        if (emptyStateMsgView.length > 0) {
            return;
        }

        var container = $('#ghx-work');
        // don't render if we don't have the element to render into
        if (container.length < 1) {
            return;
        }

        // Get the view of message
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();
        var activeSprintEmptyMsg = new ActiveSprintEmptyView(false, false, rapidViewId, true, container);

        // check of user can edit the board or not, then show the line 2
        GH.RapidViewConfig.fetchConfiguration(rapidViewId).done(function (data) {
            WorkController.trigger(WorkController.EVENT_NAMES.ON_RELOAD_EMPTY_STATE_VIEW, data.canEdit);
        });

        activeSprintEmptyMsg.handleNoMappedColumnsMode();

        WorkController.setActiveSprintEmptyMsg(activeSprintEmptyMsg);
    };

    WorkController.handleEmptyMessageInDashboard = function () {
        if (!_.isUndefined(WorkController.activeSprintEmptyMsg)) {
            WorkController.activeSprintEmptyMsg.handleOnDashboardMode();
        }
    };

    WorkController.setActiveSprintEmptyMsg = function (activeSprintEmptyMsg) {
        WorkController.activeSprintEmptyMsg = activeSprintEmptyMsg;
    };

    WorkController.collectPrintableIssues = function () {
        return {
            issues: GridDataController.collectPrintableIssues(),
            viewMode: 'work',
            boardType: GH.WorkControls.getBoardType()
        };
    };

    return WorkController;
});

AJS.namespace('GH.WorkController', null, require('jira-agile/rapid/ui/work/work-controller'));