define('jira-agile/rapid/ui/work/work-ranking', ["jira-agile/rapid/ui/work/work-controller", "jira-agile/rapid/ui/work/swimlane-view", "jira-agile/rapid/ui/work/ranking-model", "jira-agile/rapid/ajax", "jira-agile/rapid/ui/notification", "jira/util/formatter", "underscore", 'jira-agile/rapid/analytics-tracker'], function (WorkController, SwimlaneView, RankingModel, Ajax, Notification, formatter, _, AnalyticsTracker) {
    "use strict";

    /**
     * Handlers ranking operations on the work mode
     */

    var WorkRanking = {};

    WorkRanking.analytics = new AnalyticsTracker('gh.rapidboard.rankissues');

    /**
     * Re-ranks the selected issue(s) so that they are at the top of their swimlane-column.
     */
    WorkRanking.sendToTop = function (source) {
        if (!WorkController.isActive()) {
            return;
        }

        // get the currently selected issue
        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        if (!issueKey) {
            return;
        }

        // ensure the user can rank that issue
        if (!RankingModel.canUserRankIssue(issueKey)) {
            return;
        }

        // get the issue which we will be ranking before
        var firstIssueKey = RankingModel.getFirstRankableIssueKeyInColumn(issueKey);
        if (!firstIssueKey) {
            return;
        }

        // get the multi-selected issues, in currently sorted order.
        var selectedIssuesInOrder = GH.WorkSelectionController.getSelectedIssuesInOrder();

        // do analytics before we even know if they are successful
        WorkRanking.analytics.trigger('sendToTop' + (source ? source : ''), selectedIssuesInOrder.length); // SAFE

        // get a handle of the issue to select after we re-rank
        // this will be the issue that is going INTO our position - i.e. the previous issue
        var issueKeyToSelectAfterRanking = RankingModel.getBestFitSelectionAfterRank(selectedIssuesInOrder);

        // show a message once send to top has completed
        var successHandler = function successHandler() {
            // show a success message
            Notification.showSuccess(formatter.I18n.getText("gh.keyboard.shortcut.send.to.top.success"));
        };

        // do the ranking
        WorkRanking.rankIssues(selectedIssuesInOrder, null, firstIssueKey, successHandler);

        // select next issue
        if (issueKeyToSelectAfterRanking) {
            GH.WorkSelectionController.gotoIssue(issueKeyToSelectAfterRanking);
        }
    };

    /**
     * Re-ranks the selected issue(s) so that they are at the bottom of their swimlane-column.
     */
    WorkRanking.sendToBottom = function (source) {
        if (!WorkController.isActive()) {
            return;
        }

        // get the currently selected issue
        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        if (!issueKey) {
            return;
        }

        // ensure the user can rank that issue
        if (!RankingModel.canUserRankIssue(issueKey)) {
            return;
        }

        // get the issue which we will be ranking after
        var lastIssueKey = RankingModel.getLastRankableIssueKeyInColumn(issueKey);
        if (!lastIssueKey) {
            return;
        }

        // get the multi-selected issues, in currently sorted order.
        var selectedIssuesInOrder = GH.WorkSelectionController.getSelectedIssuesInOrder();

        // do analytics before we even know if they are successful
        WorkRanking.analytics.trigger('sendToBottom' + (source ? source : ''), selectedIssuesInOrder.length); // SAFE

        // get a handle of the issue to select after we re-rank
        // this will be the issue that is going INTO our position - i.e. the next issue
        var issueKeyToSelectAfterRanking = RankingModel.getBestFitSelectionAfterRank(selectedIssuesInOrder);

        // define a success handler
        var successHandler = function successHandler() {
            // show a success message
            Notification.showSuccess(formatter.I18n.getText("gh.keyboard.shortcut.send.to.bottom.success"));
        };

        // do the ranking
        WorkRanking.rankIssues(selectedIssuesInOrder, lastIssueKey, null, successHandler);

        // select the issue if we got one
        if (issueKeyToSelectAfterRanking) {
            GH.WorkSelectionController.gotoIssue(issueKeyToSelectAfterRanking);
        }
    };

    /**
     * Ranks issues on the work mode
     */
    WorkRanking.rankIssues = function (issueKeys, rankAfterKey, rankBeforeKey, successHandler) {
        // update the model
        RankingModel.reorderIssues(issueKeys, rankAfterKey, rankBeforeKey);

        // update the ui
        SwimlaneView.rerenderCellOfIssue(rankBeforeKey || rankAfterKey);

        // update the server
        var data = {
            issueKeys: issueKeys,
            rankAfterKey: rankAfterKey,
            rankBeforeKey: rankBeforeKey,
            customFieldId: RankingModel.getRankCustomFieldId()
        };

        var wrappedSuccessHandler = function wrappedSuccessHandler(response) {
            // inspect results for error response
            var errorRankResult = GH.RankController.getErrorRankResultFromResponse(response);
            if (errorRankResult) {
                Notification.showWarning(errorRankResult.errors[0]);
                WorkController.reload();
            } else if (_.isFunction(successHandler)) {
                successHandler(response);
            }
        };

        Ajax.put({
            url: '/rank',
            data: data
        }).done(wrappedSuccessHandler).fail(WorkController.reload);
    };

    return WorkRanking;
});

AJS.namespace('GH.WorkRanking', null, require('jira-agile/rapid/ui/work/work-ranking'));