define('jira-agile/rapid/ui/work/work-view', [
    "jira-agile/rapid/ui/work/work-selection-controller",
    "jira-agile/rapid/work/work-controls",
    "jira-agile/rapid/ui/work/work-controller",
    "jira-agile/rapid/ui/work/swimlane-view",
    "jira-agile/rapid/ui/work/swimline-stalker",
    "jira-agile/rapid/ui/work/issue-list-util",
    "jira-agile/rapid/ui/work/grid-data-controller",
    "jira-agile/rapid/layout",
    "jquery",
    "jira-agile/rapid/ui/kanplan/kanplan-acknowledge"
], function(
    WorkSelectionController,
    WorkControls,
    WorkController,
    SwimlaneView,
    SwimlaneStalker,
    IssueListUtil,
    GridDataController,
    Layout,
    $,
    KanPlanAcknowledge
) {
    "use strict";

    /**
     * Work Views. Handles the work view rendering
     */

    const WorkView = {};

    WorkView.isVisible = false;

    WorkView.detailViewOpen = false;

    WorkView.isRendering = false;

    WorkView.init = function() {
        // init resize handler for updating the the details view upon resize events
        $(GH).bind(Layout.EVENT_DELAYED_RESIZE, WorkView.handleResizeEvent);

        $(document).delegate('.js-view-actions-work .js-view-action-toggle-epics-lozenge', 'click', WorkController.toggleEpicsShowOnWorkboard);
    };

    /**
     * Shows the work view
     */
    WorkView.show = function() {
        WorkView.isVisible = true;

        // show the loading transition
        WorkView.showLoading();

        // create a skeleton DOM for widgets
        var skeleton = GH.tpl.workcontroller.renderSkeleton();
        $('#ghx-work').empty().append(skeleton);

        // Configure the detail view
        GH.DetailsView.setContainerSelector('#ghx-detail-view');
        GH.DetailsView.setOptions({
            canClose: true,
            showActionsCog: true,
            showSubtaskTab: true,
            showLogWorkTrigger: true,
            updateSizeHandler: WorkView.updateDetailViewSize,
            closeHandler: WorkController.closeDetailsView
        });

        if (GH.RapidBoard.State.isKanbanBoard()) {
            WorkView.showKanPlanAcknowledge();
        }
    };

    /**
     *
     * @returns {jQuery}
     */
    WorkView.getFirstColumnHeader = function () {
        return $('#ghx-column-headers .ghx-column').first();
    };

    WorkView.hide = function() {
        WorkView.isVisible = false;

        // hide the loading transition
        WorkView.hideLoading();

        // hide the detail view, as it will have to be properly positioned before displaying
        $('#ghx-detail-view').css({'display':'none'});

        // empty our DOM
        $('#ghx-work').empty();

        WorkView.hideKanPlanAcknowledge();
    };

    /**
     * Configures the detail view column displayed on the swimlane view
     * @param detailViewIssueKey the selected key or false if none selected
     */
    WorkView.configureUI = function(detailViewIssueKey) {
        // is the detail view opened?
        WorkView.detailViewOpen = !!detailViewIssueKey;

        // set the selected issue key in the detail view
        GH.DetailsView.setSelectedIssueKey(detailViewIssueKey);
    };

    /**
     * Renders the pool and detail view
     */
    WorkView.renderPoolAndDetailView = function() {
        WorkView.isRendering = true;

        // hide the spinner
        WorkView.hideLoadingPool();

        // render the swimlane view
        SwimlaneView.show();

        // update the detail view positioning (this has to be done after the swimlane is rendered!)
        WorkView.updateDetailViewVisibilityAndSize();

        // render the detail view
        WorkView.updateDetailView();

        WorkView.isRendering = false;
    };

    /**
     * Scrolls the selected issue into view
     */
    WorkView.scrollIssueToView = function() {
        // ensure the issue is visible
        var issueKey = WorkSelectionController.getSelectedIssueKey();
        if (issueKey) {
            WorkSelectionController.gotoIssue(issueKey);
        }
    };

    /**
     * Opens/collapses the detail view
     */
    WorkView.openOrCloseDetailView = function(selectedIssueKey) {
        // configure the ui
        WorkView.configureUI(selectedIssueKey);

        // update the structure
        WorkView.updateAllSizings();

        // update the detail view
        WorkView.updateDetailView();
    };

    /**
     * Updates all columns marked as fixed to ensure they fit within the content view area
     */
    // TODO: hack! I have no idea what else we should call...
    WorkView.updateFixedElementPositioning = function() {
        // if not visible don't do anything
        if (!WorkView.isVisible) {
            return;
        }

        // update Quick Filter toggle spacing
        WorkControls.resizeQuickFilters();

        GH.DetailsView.updateSize();
        SwimlaneStalker.poolStalker();
    };

    /**
     * Updates the structure and the detail view sizing
     */
    WorkView.updateAllSizings = function() {
        WorkView.updateDetailViewVisibilityAndSize();
    };

    /**
     * Updates the detail view
     */
    WorkView.updateDetailView = function() {
        if (WorkView.skipDetailsView) {
            delete WorkView.skipDetailsView;
            return;
        }
        // show/hide the detail view
        if (WorkView.detailViewOpen) {
            GH.DetailsView.show();
        } else {
            GH.DetailsView.hide();
        }

        SwimlaneStalker.poolStalker();
    };

    WorkView.skipNextDetailViewUpdate = function() {
        WorkView.skipDetailsView = true;
    };

    /**
     * Get a estimated column width
     * @return {number}
     */
    WorkView.getEstimatedColumnWidth = function() {

        var fullWidthAvailable = $('#ghx-work').width();
        var widthUsedByDetailsView = WorkView.detailViewOpen ? GH.DetailsView.minWidth : 0;
        var model = GridDataController.getModel();
        var colCount = model.getColumns().length;
        var widthUsedByGutters = (colCount + 1) * 10 + 20; // magic! each column has border-spacing (10) and the pool has a gutter (20)
        var colWidth = Math.floor((fullWidthAvailable - widthUsedByDetailsView - widthUsedByGutters) / colCount);

        return colWidth;
    };

    /**
     * Get the available width and the approximate character width for issue keys as displayed on cards,
     * for issue key truncation. These values can be null if cards are yet to be rendered.
     * Assumes this is the same value for all cards.
     */
    WorkView.getIssueKeyDimensions = function() {
        var firstVisibleIssue = $('#ghx-pool .js-issue:visible').eq(0);
        var fontSize = parseInt(firstVisibleIssue.find('.js-key-link').css('font-size'), 10);
        return {
            availableWidth: firstVisibleIssue.find('.ghx-issue-fields').width(),
            approxCharacterWidth: fontSize ? fontSize * 0.7 : null
        };
    };


    /**
     * Updates the min pool min height settings
     */
    WorkView.updateMinHeightsSettings = function() {
        if (WorkView.detailViewOpen) {
            $('#ghx-work').addClass('ghx-has-detail');
            $('#ghx-detail-view').css({'display':''});

        } else {
            // hide the view
            $('#ghx-detail-view').css({'display':'none'});

            $('#ghx-work').removeClass('ghx-has-detail');
        }
    };

    /**
     * Layout is divided into bands based on available width
     * @returns {number} - 1, 2 or 3
     */
    WorkView.getBand = function() {
        var columnWidth = WorkView.getEstimatedColumnWidth();

        if (columnWidth <= 126) {
            return 1;
        } else if (columnWidth <= 165) {
            return 2;
        } else {
            return 3;
        }
    };

    /**
     * Updates the detail view visibility, that is, shows/hides the detail view div as well as takes care of
     * keeping the detail view fixed on the screen
     */
    WorkView.updateDetailViewVisibilityAndSize = function() {
        // show if not yet visible
        var detailsContainer = GH.DetailsView.getContainer();
        if (WorkView.detailViewOpen) {

            // ensure the view is shown
            detailsContainer.css({'display':''});

            GH.DetailsView.updateSize();
        } else {
            // hide the view
            detailsContainer.css({'display':'none'});

        }

        var $pool = $('#ghx-pool');
        var band = WorkView.getBand();
        $pool.toggleClass('ghx-band-1', band === 1)
            .toggleClass('ghx-band-2', band === 2)
            .toggleClass('ghx-band-3', band === 3);

        // re-render all the issue keys after resize event
        var issueKeyDimensions = WorkView.getIssueKeyDimensions();
        var $issueKeys = $pool.find('.js-issue');
        $issueKeys.each(function(){
            var $issue = $(this);
            var issueKey = $issue.attr('data-issue-key');
            $issue.find('.js-key-link').text(IssueListUtil.buildDisplayIssueKey(issueKey, issueKeyDimensions));
        });
    };

    /**
     * Resize the column size.
     */
    // TODO: move this into the detail view See PlanView.updateDetailViewSize
    WorkView.updateDetailViewSize = function() {

        // set contents height
        var detailIssue = $('#ghx-detail-issue');
        if(detailIssue.length > 0){
            var detailIssueHeight = detailIssue.outerHeight();
            var detailHead = $('#ghx-detail-head');
            var detailHeadHeight = detailHead.outerHeight();
            var detailContents = $('#js-detail-nav-content');
            detailContents.css({'height':detailIssueHeight - detailHeadHeight - 13});// if you work out where the magic number comes from tell me
        }
    };

    /**
     * CSS Class for View Loading Transition
     */
    WorkView.showLoading = function () {
        $('body').addClass('ghx-loading-pool ghx-loading-quickfilter');
    };

    /**
     * CSS Class Removal for View Loading Transition
     */
    WorkView.hideLoading = function () {
        $('body').removeClass('ghx-loading-pool ghx-loading-quickfilter');
    };

    /**
     * CSS Class Removal for Loading Transition on QF
     */
    WorkView.hideLoadingFilter = function () {
        $('body').removeClass('ghx-loading-quickfilter');
    };

    /**
     * CSS Class for Pool Loading Transition
     */
    WorkView.showLoadingPool = function () {
        $('body').addClass('ghx-loading-pool');
        $('.ghx-throbber').spin('large');
    };

    /**
     * CSS Class Removal for Pool Loading Transition
     */
    WorkView.hideLoadingPool = function () {
        $('body').removeClass('ghx-loading-pool');
        $('.ghx-throbber').spin(false);
    };


    WorkView.handleResizeEvent = function() {
        // no-op if we are not displayed
        if (! WorkView.isVisible) {
            return;
        }

        // also skip if the model hasn't been set yet - IE triggers resize events too early
        if (! GridDataController.getModel()) {
            return;
        }

        // skip if we are rendering
        if (WorkView.isRendering) {
            return;
        }

        // update the structure
        WorkView.updateAllSizings();

        // update the swimlane view column width
        SwimlaneView.handleResizeEvent();

        // update stalker
    //    GH.SwimlaneStalker.colHeadStalker();
        SwimlaneStalker.poolStalker();

    };

    WorkView.showKanPlanAcknowledge = function () {
        KanPlanAcknowledge.show();
    };

    WorkView.hideKanPlanAcknowledge = function () {
        KanPlanAcknowledge.hide();
    };

    return WorkView;
});

AJS.namespace('GH.WorkView', null, require('jira-agile/rapid/ui/work/work-view'));