/*
 * Decompiled with CFR 0.152.
 */
package com.azure.monitor.opentelemetry.exporter.implementation.pipeline;

import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.azure.monitor.opentelemetry.exporter.implementation.ResourceAttributes;
import com.azure.monitor.opentelemetry.exporter.implementation.builders.MetricTelemetryBuilder;
import com.azure.monitor.opentelemetry.exporter.implementation.logging.OperationLogger;
import com.azure.monitor.opentelemetry.exporter.implementation.models.ContextTagKeys;
import com.azure.monitor.opentelemetry.exporter.implementation.models.TelemetryItem;
import com.azure.monitor.opentelemetry.exporter.implementation.pipeline.AppInsightsByteBufferPool;
import com.azure.monitor.opentelemetry.exporter.implementation.pipeline.ByteBufferOutputStream;
import com.azure.monitor.opentelemetry.exporter.implementation.pipeline.TelemetryPipeline;
import com.azure.monitor.opentelemetry.exporter.implementation.pipeline.TelemetryPipelineListener;
import com.azure.monitor.opentelemetry.exporter.implementation.utils.AzureMonitorMsgId;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.SerializableString;
import com.fasterxml.jackson.core.io.SerializedString;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.resources.Resource;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.GZIPOutputStream;

public class TelemetryItemExporter {
    private static final int MAX_CONCURRENT_EXPORTS = 100;
    private static final String _OTELRESOURCE_ = "_OTELRESOURCE_";
    private static final ClientLogger logger = new ClientLogger(TelemetryItemExporter.class);
    private static final OperationLogger operationLogger = new OperationLogger(TelemetryItemExporter.class, "Put export into the background (don't wait for it to return)");
    private static final ObjectMapper mapper = TelemetryItemExporter.createObjectMapper();
    private static final AppInsightsByteBufferPool byteBufferPool = new AppInsightsByteBufferPool();
    private static final OperationLogger encodeBatchOperationLogger = new OperationLogger(TelemetryItemExporter.class, "Encoding telemetry batch into json");
    private final TelemetryPipeline telemetryPipeline;
    private final TelemetryPipelineListener listener;
    private final Set<CompletableResultCode> activeExportResults = Collections.newSetFromMap(new ConcurrentHashMap());

    private static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        mapper.registerModules((Iterable)ObjectMapper.findModules((ClassLoader)TelemetryItemExporter.class.getClassLoader()));
        mapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
        return mapper;
    }

    public TelemetryItemExporter(TelemetryPipeline telemetryPipeline, TelemetryPipelineListener listener) {
        this.telemetryPipeline = telemetryPipeline;
        this.listener = listener;
    }

    public CompletableResultCode send(List<TelemetryItem> telemetryItems) {
        Map<TelemetryItemBatchKey, List<TelemetryItem>> batches = this.splitIntoBatches(telemetryItems);
        ArrayList<CompletableResultCode> resultCodeList = new ArrayList<CompletableResultCode>();
        for (Map.Entry<TelemetryItemBatchKey, List<TelemetryItem>> batch : batches.entrySet()) {
            resultCodeList.add(this.internalSendByBatch(batch.getKey(), batch.getValue()));
        }
        this.maybeAddToActiveExportResults(resultCodeList);
        return CompletableResultCode.ofAll(resultCodeList);
    }

    Map<TelemetryItemBatchKey, List<TelemetryItem>> splitIntoBatches(List<TelemetryItem> telemetryItems) {
        HashMap<TelemetryItemBatchKey, List<TelemetryItem>> groupings = new HashMap<TelemetryItemBatchKey, List<TelemetryItem>>();
        for (TelemetryItem telemetryItem : telemetryItems) {
            TelemetryItemBatchKey telemetryItemBatchKey = new TelemetryItemBatchKey(telemetryItem.getConnectionString(), telemetryItem.getResource(), telemetryItem.getResourceFromTags());
            groupings.computeIfAbsent(telemetryItemBatchKey, k -> new ArrayList()).add(telemetryItem);
        }
        return groupings;
    }

    private void maybeAddToActiveExportResults(List<CompletableResultCode> results) {
        if (this.activeExportResults.size() >= 100) {
            operationLogger.recordFailure("Hit max 100 active concurrent requests", AzureMonitorMsgId.TELEMETRY_ITEM_EXPORTER_ERROR);
        }
        operationLogger.recordSuccess();
        this.activeExportResults.addAll(results);
        for (CompletableResultCode result : results) {
            result.whenComplete(() -> this.activeExportResults.remove(result));
        }
    }

    public CompletableResultCode flush() {
        return CompletableResultCode.ofAll(this.activeExportResults);
    }

    public CompletableResultCode shutdown() {
        return this.listener.shutdown();
    }

    CompletableResultCode internalSendByBatch(TelemetryItemBatchKey telemetryItemBatchKey, List<TelemetryItem> telemetryItems) {
        List<ByteBuffer> byteBuffers;
        if (!"Statsbeat".equals(telemetryItems.get(0).getName())) {
            telemetryItems.add(0, this.createOtelResourceMetric(telemetryItemBatchKey));
        }
        try {
            byteBuffers = this.encode(telemetryItems);
            encodeBatchOperationLogger.recordSuccess();
        }
        catch (Throwable t) {
            encodeBatchOperationLogger.recordFailure(t.getMessage(), t);
            return CompletableResultCode.ofFailure();
        }
        return this.telemetryPipeline.send(byteBuffers, telemetryItemBatchKey.connectionString, this.listener);
    }

    private TelemetryItem createOtelResourceMetric(TelemetryItemBatchKey telemetryItemBatchKey) {
        String internalSdkVersion;
        String roleInstance;
        MetricTelemetryBuilder builder = MetricTelemetryBuilder.create(_OTELRESOURCE_, 0.0);
        builder.setConnectionString(telemetryItemBatchKey.connectionString);
        telemetryItemBatchKey.resource.getAttributes().forEach((k, v) -> builder.addProperty(k.getKey(), v.toString()));
        String roleName = (String)telemetryItemBatchKey.resourceFromTags.get(ContextTagKeys.AI_CLOUD_ROLE.toString());
        if (roleName != null) {
            builder.addProperty(ResourceAttributes.SERVICE_NAME.getKey(), roleName);
            builder.addTag(ContextTagKeys.AI_CLOUD_ROLE.toString(), roleName);
        }
        if ((roleInstance = (String)telemetryItemBatchKey.resourceFromTags.get(ContextTagKeys.AI_CLOUD_ROLE_INSTANCE.toString())) != null) {
            builder.addProperty(ResourceAttributes.SERVICE_INSTANCE_ID.getKey(), roleInstance);
            builder.addTag(ContextTagKeys.AI_CLOUD_ROLE_INSTANCE.toString(), roleInstance);
        }
        if ((internalSdkVersion = (String)telemetryItemBatchKey.resourceFromTags.get(ContextTagKeys.AI_INTERNAL_SDK_VERSION.toString())) != null) {
            builder.addTag(ContextTagKeys.AI_INTERNAL_SDK_VERSION.toString(), internalSdkVersion);
        }
        return builder.build();
    }

    List<ByteBuffer> encode(List<TelemetryItem> telemetryItems) throws IOException {
        JsonGenerator jg;
        if (logger.canLogAtLevel(LogLevel.VERBOSE)) {
            StringWriter debug = new StringWriter();
            jg = mapper.createGenerator((Writer)debug);
            try {
                TelemetryItemExporter.writeTelemetryItems(jg, telemetryItems);
            }
            finally {
                if (jg != null) {
                    jg.close();
                }
            }
            logger.verbose("sending telemetry to ingestion service:{}{}", new Object[]{System.lineSeparator(), debug});
        }
        ByteBufferOutputStream out = new ByteBufferOutputStream(byteBufferPool);
        try {
            jg = mapper.createGenerator((OutputStream)new GZIPOutputStream(out));
            try {
                TelemetryItemExporter.writeTelemetryItems(jg, telemetryItems);
            }
            finally {
                if (jg != null) {
                    jg.close();
                }
            }
        }
        catch (IOException e) {
            byteBufferPool.offer(out.getByteBuffers());
            throw e;
        }
        out.close();
        List<ByteBuffer> byteBuffers = out.getByteBuffers();
        for (ByteBuffer byteBuffer : byteBuffers) {
            byteBuffer.flip();
        }
        return byteBuffers;
    }

    private static void writeTelemetryItems(JsonGenerator jg, List<TelemetryItem> telemetryItems) throws IOException {
        jg.setRootValueSeparator((SerializableString)new SerializedString("\n"));
        for (TelemetryItem telemetryItem : telemetryItems) {
            mapper.writeValue(jg, (Object)telemetryItem);
        }
    }

    private static class TelemetryItemBatchKey {
        private final String connectionString;
        private final Resource resource;
        private final Map<String, String> resourceFromTags;

        private TelemetryItemBatchKey(String connectionString, Resource resource, Map<String, String> resourceFromTags) {
            this.connectionString = connectionString;
            this.resource = resource;
            this.resourceFromTags = resourceFromTags;
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || this.getClass() != other.getClass()) {
                return false;
            }
            TelemetryItemBatchKey that = (TelemetryItemBatchKey)other;
            return Objects.equals(this.connectionString, that.connectionString) && Objects.equals(this.resource, that.resource) && Objects.equals(this.resourceFromTags, that.resourceFromTags);
        }

        public int hashCode() {
            return Objects.hash(this.connectionString, this.resource, this.resourceFromTags);
        }
    }
}

