/*
 * Copyright (c) 2012 SureLogic, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.surelogic;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Declares that object referenced by the return value, receiver, or formal
 * parameter is only partially initialized. That is, some of the fields of the
 * class may hold <code>null</code> values even though the field is annotated to
 * be {@link NonNull}. Furthermore, the return value, receiver, or formal
 * parameter is itself non-<code>null</code>.
 * 
 * <p>
 * The extent to which the referenced object is initialized is described by the
 * {@link #upTo() upTo} attribute:
 * 
 * <dl>
 * <dt>&#064;Raw(upTo="*")
 * <dd>The object must be assumed to be completely uninitialized. Any
 * {@link NonNull} field of the object may in fact be <code>null</code>. This is
 * the default value of the {@link #upTo upTo} attribute, and there is never any
 * reason to state it explicitly. That is, <code>&#064;Raw</code> is preferred
 * over <code>&#064;Raw(upTo="*")</code>.
 * 
 * <dt>&#064;Raw(upTo="<i>qualified-type-name</i>")
 * <dd>The object may be assumed to initialized up to the named ancestor type.
 * That is, the constructors for each ancestor type from
 * <code>java.lang.Object</code> down to and including the named type have been
 * successfully invoked. Any {@link NonNull} field in the object declared in
 * types that are descendants of the named type may in fact be <code>null</code>.
 * </dl>
 * 
 * <p>When the annotation appears on a method, the {@link #value() value} attribute
 * is used to distinguish whether the return value or receiver is being 
 * annotated.
 * 
 * <p><em>Need to describe the <code>static(type)</code> value for upTo.</em>
 * 
 * @see NonNull
 * @see Nullable
 * @see Raws
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target({ ElementType.METHOD, ElementType.CONSTRUCTOR, ElementType.PARAMETER })
public @interface Raw {
  String upTo() default "*";

  String value() default "";
}
