package com.github.ajalt.clikt.parameters.groups

import com.github.ajalt.clikt.core.BadParameterValue
import com.github.ajalt.clikt.core.BaseCliktCommand
import com.github.ajalt.clikt.core.Context
import com.github.ajalt.clikt.core.MissingOption
import com.github.ajalt.clikt.parameters.internal.NullableLateinit
import com.github.ajalt.clikt.parameters.options.Option
import com.github.ajalt.clikt.parameters.options.OptionDelegate
import com.github.ajalt.clikt.parameters.options.RawOption
import com.github.ajalt.clikt.parameters.options.switch
import com.github.ajalt.clikt.parameters.types.choice
import com.github.ajalt.clikt.parsers.OptionInvocation
import kotlin.properties.ReadOnlyProperty
import kotlin.reflect.KProperty

class ChoiceGroup<GroupT : OptionGroup, OutT> internal constructor(
    internal val option: OptionDelegate<String?>,
    internal val groups: Map<String, GroupT>,
    internal val transform: (GroupT?) -> OutT,
) : ParameterGroupDelegate<OutT> {
    override val groupName: String? = null
    override val groupHelp: String? = null
    private var value: OutT by NullableLateinit("Cannot read from option delegate before parsing command line")
    private var chosenGroup: OptionGroup? = null

    init {
        require(groups.none { it.value.options.any { o -> o.eager } }) {
            "eager options are not allowed in choice and switch option groups"
        }
    }

    override fun provideDelegate(
        thisRef: BaseCliktCommand<*>,
        property: KProperty<*>,
    ): ReadOnlyProperty<BaseCliktCommand<*>, OutT> {
        option.provideDelegate(thisRef, property) // infer the option name and register it
        thisRef.registerOptionGroup(this)
        for ((_, group) in groups) {
            for (option in group.options) {
                option.parameterGroup = this
                option.groupName = group.groupName
                thisRef.registerOption(option)
            }
        }
        return this
    }

    override fun getValue(thisRef: BaseCliktCommand<*>, property: KProperty<*>): OutT = value

    override fun finalize(
        context: Context,
        invocationsByOption: Map<Option, List<OptionInvocation>>,
    ) {
        val key = option.value
        if (key == null) {
            value = transform(null)
            // Finalize the group so that default groups have their options finalized
            (value as? OptionGroup)?.let { g ->
                g.finalize(context, invocationsByOption.filterKeys { it in g.options })
                chosenGroup = g
            }
            return
        }

        val group = groups[key] ?: throw BadParameterValue(
            context.localization.invalidGroupChoice(key, groups.keys.toList()),
            option,
        )
        group.finalize(context, invocationsByOption.filterKeys { it in group.options })
        chosenGroup = group
        value = transform(group)
    }

    override fun postValidate(context: Context) {
        chosenGroup?.options?.forEach { it.postValidate(context) }
    }
}

/**
 * Convert the option to an option group based on a fixed set of values.
 *
 * ### Example:
 *
 * ```
 * option().groupChoice(mapOf("foo" to FooOptionGroup(), "bar" to BarOptionGroup()))
 * ```
 *
 * @see com.github.ajalt.clikt.parameters.types.choice
 */
fun <T : OptionGroup> RawOption.groupChoice(choices: Map<String, T>): ChoiceGroup<T, T?> {
    return ChoiceGroup(choice(choices.mapValues { it.key }), choices) { it }
}

/**
 * Convert the option to an option group based on a fixed set of values.
 *
 * ### Example:
 *
 * ```
 * option().groupChoice("foo" to FooOptionGroup(), "bar" to BarOptionGroup())
 * ```
 *
 * @see com.github.ajalt.clikt.parameters.types.choice
 */
fun <T : OptionGroup> RawOption.groupChoice(vararg choices: Pair<String, T>): ChoiceGroup<T, T?> {
    return groupChoice(choices.toMap())
}

/**
 * If a [groupChoice] or [groupSwitch] option is not called on the command line, throw a
 * [MissingOption] exception.
 *
 * ### Example:
 *
 * ```
 * option().groupChoice("foo" to FooOptionGroup(), "bar" to BarOptionGroup()).required()
 * ```
 */
fun <T : OptionGroup> ChoiceGroup<T, T?>.required(): ChoiceGroup<T, T> {
    return ChoiceGroup(option, groups) { it ?: throw MissingOption(option) }
}

/**
 * Convert the option into a set of flags that each map to an option group.
 *
 * ### Example:
 *
 * ```
 * option().groupSwitch(mapOf("--foo" to FooOptionGroup(), "--bar" to BarOptionGroup()))
 * ```
 */
fun <T : OptionGroup> RawOption.groupSwitch(choices: Map<String, T>): ChoiceGroup<T, T?> {
    return ChoiceGroup(switch(choices.mapValues { it.key }), choices) { it }
}

/**
 * Convert the option into a set of flags that each map to an option group.
 *
 * ### Example:
 *
 * ```
 * option().groupSwitch("--foo" to FooOptionGroup(), "--bar" to BarOptionGroup())
 * ```
 */
fun <T : OptionGroup> RawOption.groupSwitch(vararg choices: Pair<String, T>): ChoiceGroup<T, T?> {
    return groupSwitch(choices.toMap())
}

/**
 * If a [groupChoice] or [groupSwitch] option is not called on the command line, use the value of
 * the group with a switch or choice [name].
 *
 * ### Example:
 *
 * ```
 * option().groupChoice("foo" to FooOptionGroup(), "bar" to BarOptionGroup()).defaultByName("foo")
 * option().groupSwitch("--foo" to FooOptionGroup(), "--bar" to BarOptionGroup()).defaultByName("--bar")
 * ```
 *
 * @throws IllegalArgumentException if [name] is not one of the option's choice/switch names.
 */
fun <T : OptionGroup> ChoiceGroup<T, T?>.defaultByName(name: String): ChoiceGroup<T, T> {
    require(name in groups) { "invalid default name $name (must be one of ${groups.keys})" }
    return ChoiceGroup(option, groups) { it ?: groups.getValue(name) }
}
