/*
 * Decompiled with CFR 0.152.
 */
package org.tinyradius.dictionary;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import org.tinyradius.attribute.AttributeType;
import org.tinyradius.dictionary.MemoryDictionary;
import org.tinyradius.dictionary.WritableDictionary;

public class DictionaryParser {
    private final ResourceResolver resourceResolver;

    private DictionaryParser(ResourceResolver resourceResolver) {
        this.resourceResolver = resourceResolver;
    }

    public static DictionaryParser newClasspathParser() {
        return new DictionaryParser(new ClasspathResourceResolver());
    }

    public static DictionaryParser newFileParser() {
        return new DictionaryParser(new FileResourceResolver());
    }

    public WritableDictionary parseDictionary(String resource) throws IOException {
        MemoryDictionary d = new MemoryDictionary();
        this.parseDictionary(d, resource);
        return d;
    }

    public void parseDictionary(WritableDictionary dictionary, String resource) throws IOException {
        try (InputStream inputStream = this.resourceResolver.openStream(resource);
             BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));){
            String line;
            int lineNum = -1;
            while ((line = in.readLine()) != null) {
                String[] tokens;
                ++lineNum;
                if ((line = line.trim()).startsWith("#") || line.isEmpty() || (tokens = line.split("\\s+")).length == 0) continue;
                this.parseLine(dictionary, tokens, lineNum, resource);
            }
        }
    }

    private void parseLine(WritableDictionary dictionary, String[] tokens, int lineNum, String resource) throws IOException {
        switch (tokens[0].toUpperCase()) {
            case "ATTRIBUTE": {
                this.parseAttributeLine(dictionary, tokens, lineNum);
                break;
            }
            case "VALUE": {
                this.parseValueLine(dictionary, tokens, lineNum);
                break;
            }
            case "$INCLUDE": {
                this.includeDictionaryFile(dictionary, tokens, lineNum, resource);
                break;
            }
            case "VENDORATTR": {
                this.parseVendorAttributeLine(dictionary, tokens, lineNum);
                break;
            }
            case "VENDOR": {
                this.parseVendorLine(dictionary, tokens, lineNum);
                break;
            }
            default: {
                throw new IOException("Unknown line type: " + tokens[0] + " line: " + lineNum);
            }
        }
    }

    private void parseAttributeLine(WritableDictionary dictionary, String[] tok, int lineNum) throws IOException {
        if (tok.length != 4) {
            throw new IOException("Attribute parse error on line " + lineNum + ", " + Arrays.toString(tok));
        }
        String name = tok[1];
        int code = Integer.parseInt(tok[2]);
        String typeStr = tok[3];
        dictionary.addAttributeType(new AttributeType(code, name, typeStr));
    }

    private void parseValueLine(WritableDictionary dictionary, String[] tok, int lineNum) throws IOException {
        if (tok.length != 4) {
            throw new IOException("Value parse error on line " + lineNum + ": " + Arrays.toString(tok));
        }
        String typeName = tok[1];
        String enumName = tok[2];
        String valStr = tok[3];
        AttributeType at = dictionary.getAttributeTypeByName(typeName);
        if (at == null) {
            throw new IOException("Unknown attribute type: " + typeName + ", line: " + lineNum);
        }
        at.addEnumerationValue(Integer.parseInt(valStr), enumName);
    }

    private void parseVendorAttributeLine(WritableDictionary dictionary, String[] tok, int lineNum) throws IOException {
        if (tok.length != 5) {
            throw new IOException("Vendor Attribute parse error on line " + lineNum + ": " + Arrays.toString(tok));
        }
        int vendor = Integer.parseInt(tok[1]);
        String name = tok[2];
        int code = Integer.parseInt(tok[3]);
        String typeStr = tok[4];
        dictionary.addAttributeType(new AttributeType(vendor, code, name, typeStr));
    }

    private void parseVendorLine(WritableDictionary dictionary, String[] tok, int lineNum) throws IOException {
        if (tok.length != 3) {
            throw new IOException("Vendor parse error on line " + lineNum + ": " + Arrays.toString(tok));
        }
        int vendorId = Integer.parseInt(tok[1]);
        String vendorName = tok[2];
        dictionary.addVendor(vendorId, vendorName);
    }

    private void includeDictionaryFile(WritableDictionary dictionary, String[] tok, int lineNum, String currentResource) throws IOException {
        if (tok.length != 2) {
            throw new IOException("Dictionary include parse error on line " + lineNum + ": " + Arrays.toString(tok));
        }
        String includeFile = tok[1];
        String nextResource = this.resourceResolver.resolve(currentResource, includeFile);
        if (nextResource.isEmpty()) {
            throw new IOException("Included file '" + includeFile + "' was not found, line " + lineNum + ", " + currentResource);
        }
        this.parseDictionary(dictionary, nextResource);
    }

    private static class ClasspathResourceResolver
    implements ResourceResolver {
        private ClasspathResourceResolver() {
        }

        @Override
        public String resolve(String currentResource, String nextResource) {
            String path = Paths.get(currentResource, new String[0]).getParent().resolve(nextResource).toString();
            return this.getClass().getClassLoader().getResource(path) != null ? path : "";
        }

        @Override
        public InputStream openStream(String resource) throws IOException {
            InputStream stream = this.getClass().getClassLoader().getResourceAsStream(resource);
            if (stream != null) {
                return stream;
            }
            throw new IOException("could not open stream, classpath resource not found: " + resource);
        }
    }

    private static class FileResourceResolver
    implements ResourceResolver {
        private FileResourceResolver() {
        }

        @Override
        public String resolve(String currentResource, String nextResource) {
            Path path = Paths.get(currentResource, new String[0]).getParent().resolve(nextResource);
            return Files.exists(path, new LinkOption[0]) ? path.toString() : "";
        }

        @Override
        public InputStream openStream(String resource) throws IOException {
            Path path = Paths.get(resource, new String[0]);
            if (Files.exists(path, new LinkOption[0])) {
                return Files.newInputStream(path, new OpenOption[0]);
            }
            throw new IOException("could not open stream, file not found: " + resource);
        }
    }

    public static interface ResourceResolver {
        public String resolve(String var1, String var2);

        public InputStream openStream(String var1) throws IOException;
    }
}

