// Copyright 2011 Google Inc. All Rights Reserved.

package com.google.appengine.tools.mapreduce;

import static com.google.common.base.Preconditions.checkNotNull;

import com.google.appengine.tools.mapreduce.reducers.NoReducer;

import java.io.Serializable;

/**
 * Specification for a MapReduce job.  The values here affect what computation
 * is performed in the MapReduce and its side-effects, but not how it is
 * executed; see {@link MapReduceSettings} for that.
 *
 * @author ohler@google.com (Christian Ohler)
 *
 * @param <I> type of input values
 * @param <K> type of intermediate keys
 * @param <V> type of intermediate values
 * @param <O> type of output values
 * @param <R> type of result returned by the {@link Output}
 */
public final class MapReduceSpecification<I, K, V, O, R> implements Serializable {
  private static final long serialVersionUID = 4773121986957455423L;

  /**
   * Creates a new {@code MapReduceSpecification} with the given parameters.
   *
   * @param jobName descriptive name for the job (to be displayed to humans,
   *     need not be a unique id)
   * @param input specifies what input the mapper should process
   * @param mapper processes the input and optionally generates intermediate
   *     key-value pairs to be processed by {@code reducer}
   * @param intermediateKeyMarshaller turns intermediate keys into bytes and
   *     back.  See {@link Marshallers} for defaults.  Use
   *     {@link Marshallers#getVoidMarshaller} if the mapper generates no
   *     intermediate output
   * @param intermediateValueMarshaller turns intermediate values into bytes and
   *     back.  See {@link Marshallers} for defaults.  Use
   *     {@link Marshallers#getVoidMarshaller} if the mapper generates no
   *     intermediate output
   * @param reducer processes the intermediate key-value pairs generated by
   *     {@code mapper} and optionally writes output to {@code output}.  Use
   *     {@link NoReducer} if the mapper generates no intermediate output
   * @param output specifies what to do with the output values generated by
   *     {@code} reducer, and produces the result value of the
   *     {@link MapReduceJob}
   */
  public static <I, K, V, O, R>
      MapReduceSpecification<I, K, V, O, R> of(
          String jobName,
          Input<I> input,
          Mapper<I, K, V> mapper,
          Marshaller<K> intermediateKeyMarshaller,
          Marshaller<V> intermediateValueMarshaller,
          Reducer<K, V, O> reducer,
          Output<O, R> output) {
    return new MapReduceSpecification<I, K, V, O, R>(
        jobName, input, mapper, intermediateKeyMarshaller, intermediateValueMarshaller,
        reducer, output);
  }

  // User-specified name for the job.  Need not be unique.
  private String jobName;
  private Input<I> input;
  private Mapper<I, K, V> mapper;
  private Marshaller<K> intermediateKeyMarshaller;
  private Marshaller<V> intermediateValueMarshaller;
  private Reducer<K, V, O> reducer;
  private Output<O, R> output;
  // TODO(ohler): Add these features.  It's tricky to do it in a way that avoids
  // unnecessary unmarshalling in the default case (byte comparison) though.
  ///*Nullable*/ private Comparator<K> intermediateKeyComparator = null;
  ///*Nullable*/ private ReduceShardFunction<K> reduceShardFunction = null;

  private MapReduceSpecification(String jobName,
      Input<I> input,
      Mapper<I, K, V> mapper,
      Marshaller<K> intermediateKeyMarshaller,
      Marshaller<V> intermediateValueMarshaller,
      Reducer<K, V, O> reducer,
      Output<O, R> output) {
    this.jobName = checkNotNull(jobName, "Null jobName");
    this.input = checkNotNull(input, "Null input");
    this.mapper = checkNotNull(mapper, "Null mapper");
    this.intermediateKeyMarshaller =
        checkNotNull(intermediateKeyMarshaller, "Null intermediateKeyMarshaller");
    this.intermediateValueMarshaller =
        checkNotNull(intermediateValueMarshaller, "Null intermediateValueMarshaller");
    this.reducer = checkNotNull(reducer, "Null reducer");
    this.output = checkNotNull(output, "Null output");
  }

  public String getJobName() {
    return jobName;
  }

  public MapReduceSpecification<I, K, V, O, R> setJobName(String jobName) {
    this.jobName = checkNotNull(jobName, "Null jobName");
    return this;
  }

  public Input<I> getInput() {
    return input;
  }

  public MapReduceSpecification<I, K, V, O, R> setInput(Input<I> input) {
    this.input = checkNotNull(input, "Null input");
    return this;
  }

  public Mapper<I, K, V> getMapper() {
    return mapper;
  }

  public MapReduceSpecification<I, K, V, O, R> setMapper(Mapper<I, K, V> mapper) {
    this.mapper = checkNotNull(mapper, "Null mapper");
    return this;
  }

  public Marshaller<K> getIntermediateKeyMarshaller() {
    return intermediateKeyMarshaller;
  }

  public MapReduceSpecification<I, K, V, O, R> setIntermediateKeyMarshaller(
      Marshaller<K> intermediateKeyMarshaller) {
    this.intermediateKeyMarshaller =
        checkNotNull(intermediateKeyMarshaller, "Null intermediateKeyMarshaller");
    return this;
  }

  public Marshaller<V> getIntermediateValueMarshaller() {
    return intermediateValueMarshaller;
  }

  public MapReduceSpecification<I, K, V, O, R> setIntermediateValueMarshaller(
      Marshaller<V> intermediateValueMarshaller) {
    this.intermediateValueMarshaller =
        checkNotNull(intermediateValueMarshaller, "Null intermediateValueMarshaller");
    return this;
  }

  public Reducer<K, V, O> getReducer() {
    return reducer;
  }

  public MapReduceSpecification<I, K, V, O, R> setReducer(Reducer<K, V, O> reducer) {
    this.reducer = checkNotNull(reducer, "Null reducer");
    return this;
  }

  public Output<O, R> getOutput() {
    return output;
  }

  public MapReduceSpecification<I, K, V, O, R> setOutput(Output<O, R> output) {
    this.output = checkNotNull(output, "Null output");
    return this;
  }

  @Override public String toString() {
    return getClass().getSimpleName() + "("
        + jobName + ", "
        + input + ", "
        + mapper + ", "
        + reducer + ", "
        + output
        + ")";
  }

}
