/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.isTopLevel
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirMemberDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.expectForActual
import ksp.org.jetbrains.kotlin.fir.declarations.hasAnnotation
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isActual
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.isEnabled
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.resolve.multiplatform.ExpectActualMatchingCompatibility

object FirExpectRefinementChecker : FirBasicDeclarationChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirDeclaration) {
        val hasExpectRefinementAnnotation =
            declaration.hasAnnotation(StandardClassIds.Annotations.ExpectRefinement, context.session)
        val isExpect = declaration is FirMemberDeclaration && declaration.isExpect
        if (hasExpectRefinementAnnotation && (!isExpect || !context.isTopLevel)) {
            reporter.reportOn(
                declaration.source,
                FirErrors.EXPECT_REFINEMENT_ANNOTATION_WRONG_TARGET
            )
            return
        }
        if (declaration !is FirMemberDeclaration) return
        val matchingData = declaration.symbol.expectForActual.orEmpty()
        val matchedWithAnotherExpect = matchingData.contains(ExpectActualMatchingCompatibility.MatchedSuccessfully)
        if (matchedWithAnotherExpect && declaration.isExpect && !declaration.isActual && context.isTopLevel) {
            if (!hasExpectRefinementAnnotation) {
                reporter.reportOn(
                    declaration.source,
                    FirErrors.EXPECT_REFINEMENT_ANNOTATION_MISSING
                )
            }
            if (!LanguageFeature.ExpectRefinement.isEnabled()) {
                reporter.reportOn(
                    declaration.source,
                    FirErrors.UNSUPPORTED_FEATURE,
                    LanguageFeature.ExpectRefinement to context.languageVersionSettings
                )
            }
        }
        if (!matchedWithAnotherExpect && hasExpectRefinementAnnotation) {
            reporter.reportOn(
                declaration.source,
                FirErrors.ACTUAL_WITHOUT_EXPECT,
                declaration.symbol,
                matchingData
            )
        }
    }
}
