/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.type

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.FirFunctionTypeParameter
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.checkUpperBoundViolated
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.declarations.FirCallableDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirClass
import ksp.org.jetbrains.kotlin.fir.declarations.FirTypeParameter
import ksp.org.jetbrains.kotlin.fir.expressions.FirTypeOperatorCall
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirTypeAliasSymbol
import ksp.org.jetbrains.kotlin.fir.types.FirFunctionTypeRef
import ksp.org.jetbrains.kotlin.fir.types.FirResolvedTypeRef
import ksp.org.jetbrains.kotlin.fir.types.FirTypeProjectionWithVariance
import ksp.org.jetbrains.kotlin.fir.types.FirUserTypeRef

object FirUpperBoundViolatedTypeChecker : FirResolvedTypeRefChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(typeRef: FirResolvedTypeRef) {
        val container = context.containingElements.dropLast(1).lastOrNull()
        val isUserTypeWithoutArguments = (typeRef.delegatedTypeRef as? FirUserTypeRef)?.qualifier?.lastOrNull()
            .let { it != null && it.typeArgumentList.typeArguments.isEmpty() }
        val isInTypeOperatorCall = context.containingElements.dropLast(1).lastOrNull() is FirTypeOperatorCall
        val isBareType = isInTypeOperatorCall && isUserTypeWithoutArguments
        val isInParameterBoundsOfNonClass = context.containingElements.dropLast(1).lastOrNull() is FirTypeParameter
                && context.containingElements.dropLast(2).lastOrNull() !is FirClass

        if (
            container is FirCallableDeclaration && typeRef.source?.kind is KtFakeSourceElementKind ||
            // `checkUpperBoundViolated()` will fully-expand and traverse the arguments on its own -
            // it must do so because bound violations within typealiases are use-site-dependent.
            // Kotlin doesn't support type parameter bounds for typealiases, meaning we can't prohibit
            // `typealias TA<F> = NotNullBox<F>` as there would be no workaround.
            container is FirTypeProjectionWithVariance || container is FirFunctionTypeParameter || container is FirFunctionTypeRef ||
            // Otherwise, we'd fail on bare casts like `it is FirClassSymbol` for `it: FirBasedSymbol<FirDeclaration>`
            // because the compiler infers `FirClassSymbol<FirDeclaration>` for the RHS.
            isBareType
        ) {
            return
        }

        checkUpperBoundViolated(
            typeRef,
            isIgnoreTypeParameters = context.containingDeclarations.lastOrNull() is FirTypeAliasSymbol,
            isInsideTypeOperatorOrParameterBounds = isInTypeOperatorCall || isInParameterBoundsOfNonClass,
        )
    }
}