/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.resolve.transformers.plugin

import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.FirClassLikeDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirFile
import ksp.org.jetbrains.kotlin.fir.declarations.FirRegularClass
import ksp.org.jetbrains.kotlin.fir.declarations.FirResolvePhase
import ksp.org.jetbrains.kotlin.fir.declarations.utils.classId
import ksp.org.jetbrains.kotlin.fir.expressions.FirStatement
import ksp.org.jetbrains.kotlin.fir.extensions.FirSwitchableExtensionDeclarationsSymbolProvider
import ksp.org.jetbrains.kotlin.fir.extensions.generatedDeclarationsSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.resolve.transformers.FirTransformerBasedResolveProcessor
import ksp.org.jetbrains.kotlin.fir.scopes.impl.nestedClassifierScope
import ksp.org.jetbrains.kotlin.fir.scopes.processClassifiersByName
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.utils.exceptions.withFirEntry
import ksp.org.jetbrains.kotlin.fir.utils.exceptions.withFirSymbolEntry
import ksp.org.jetbrains.kotlin.fir.visitors.FirTransformer
import ksp.org.jetbrains.kotlin.fir.visitors.transformSingle
import ksp.org.jetbrains.kotlin.fir.withFileAnalysisExceptionWrapping
import ksp.org.jetbrains.kotlin.name.SpecialNames.DEFAULT_NAME_FOR_COMPANION_OBJECT
import ksp.org.jetbrains.kotlin.utils.exceptions.errorWithAttachment

class FirCompanionGenerationProcessor(
    session: FirSession,
    scopeSession: ScopeSession
) : FirTransformerBasedResolveProcessor(session, scopeSession, FirResolvePhase.COMPANION_GENERATION) {
    override val transformer: FirTransformer<Nothing?> = FirCompanionGenerationTransformer(session)
}

class FirCompanionGenerationTransformer(val session: FirSession) : FirTransformer<Nothing?>() {
    private val generatedDeclarationProvider: FirSwitchableExtensionDeclarationsSymbolProvider? =
        session.generatedDeclarationsSymbolProvider

    override fun <E : FirElement> transformElement(element: E, data: Nothing?): E {
        return element
    }

    override fun transformFile(file: FirFile, data: Nothing?): FirFile {
        // I don't want to use laziness here to prevent possible multi-threading problems
        if (generatedDeclarationProvider == null) return file
        return withFileAnalysisExceptionWrapping(file) {
            file.transformDeclarations(this, data)
        }
    }

    override fun transformRegularClass(regularClass: FirRegularClass, data: Nothing?): FirStatement {
        generateAndUpdateCompanion(regularClass)
        return regularClass.transformDeclarations(this, data)
    }

    fun generateAndUpdateCompanion(regularClass: FirRegularClass) {
        val companionSymbol = generateCompanion(regularClass)
        if (companionSymbol != null) {
            regularClass.replaceCompanionObjectSymbol(companionSymbol)
        }
    }

    private fun generateCompanion(regularClass: FirRegularClass): FirRegularClassSymbol? {
        if (generatedDeclarationProvider == null) return null
        val generatedCompanion = if (regularClass.isLocal) {
            var result: FirClassLikeSymbol<*>? = null
            session.nestedClassifierScope(regularClass)?.processClassifiersByName(DEFAULT_NAME_FOR_COMPANION_OBJECT) {
                if (it is FirClassLikeSymbol<*> && it.origin.generated) {
                    result = it
                }
            }

            result
        } else {
            val companionClassId = regularClass.classId.createNestedClassId(DEFAULT_NAME_FOR_COMPANION_OBJECT)
            generatedDeclarationProvider.getClassLikeSymbolByClassId(companionClassId)?.takeIf { it.origin.generated }
        }

        return when (generatedCompanion) {
            null -> null

            is FirRegularClassSymbol -> when {
                regularClass.companionObjectSymbol != null -> errorWithAttachment("Plugin generated duplicated companion object: ${generatedCompanion.origin::class.simpleName}") {
                    withEntry("origin", generatedCompanion.origin.toString())
                    withFirSymbolEntry("generatedCompanion", generatedCompanion)
                    withFirEntry("regularClass", regularClass)
                }

                else -> generatedCompanion
            }

            else -> errorWithAttachment("Plugin generated non regular class as companion object: ${generatedCompanion.origin::class.simpleName}") {
                withEntry("origin", generatedCompanion.origin.toString())
                withFirSymbolEntry("generatedCompanion", generatedCompanion)
                withFirEntry("regularClass", regularClass)
            }
        }
    }
}

fun <F : FirClassLikeDeclaration> F.runCompanionGenerationPhaseForLocalClass(session: FirSession): F {
    val transformer = FirCompanionGenerationTransformer(session)
    return this.transformSingle(transformer, null)
}
