/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.descriptors.Modality
import ksp.org.jetbrains.kotlin.descriptors.Visibilities
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.KtDiagnosticFactory0
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.findClosestClassOrObject
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirBackingField
import ksp.org.jetbrains.kotlin.fir.declarations.impl.FirDefaultPropertyBackingField
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isAbstract
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExtension
import ksp.org.jetbrains.kotlin.fir.declarations.utils.modality
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility

object FirExplicitBackingFieldForbiddenChecker : FirBackingFieldChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirBackingField) {
        if (declaration is FirDefaultPropertyBackingField) {
            return
        }

        if (declaration.propertySymbol.modality != Modality.FINAL) {
            reporter.reportOn(declaration.source, getProperDiagnostic(declaration))
        }

        if (declaration.propertySymbol.isExtension) {
            reporter.reportOn(declaration.source, FirErrors.EXPLICIT_BACKING_FIELD_IN_EXTENSION)
        }

        if (declaration.propertySymbol.isExpect) {
            reporter.reportOn(declaration.propertySymbol.source, FirErrors.EXPECT_PROPERTY_WITH_EXPLICIT_BACKING_FIELD)
        }

        if (Visibilities.isPrivate(declaration.propertySymbol.visibility)) {
            reporter.reportOn(declaration.propertySymbol.source, FirErrors.EXPLICIT_FIELD_VISIBILITY_MUST_BE_LESS_PERMISSIVE)
        }
    }

    context(context: CheckerContext)
    private fun getProperDiagnostic(declaration: FirBackingField): KtDiagnosticFactory0 {
        return when {
            context.findClosestClassOrObject()?.classKind == ClassKind.INTERFACE -> FirErrors.EXPLICIT_BACKING_FIELD_IN_INTERFACE
            declaration.propertySymbol.isAbstract -> FirErrors.EXPLICIT_BACKING_FIELD_IN_ABSTRACT_PROPERTY
            else -> FirErrors.NON_FINAL_PROPERTY_WITH_EXPLICIT_BACKING_FIELD
        }
    }
}
