/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.descriptors.EffectiveVisibility
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.containingClassLookupTag
import ksp.org.jetbrains.kotlin.fir.declarations.utils.effectiveVisibility
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.getContainingClassLookupTag
import ksp.org.jetbrains.kotlin.fir.resolve.fullyExpandedType
import ksp.org.jetbrains.kotlin.fir.resolve.toClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.transformers.publishedApiEffectiveVisibility
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.toEffectiveVisibility
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.forEachType

object FirInlineExposedLessVisibleTypeQualifiedAccessChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Platform) {
    context(c: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirQualifiedAccessExpression) {
        val inlineFunctionBodyContext = c.inlineFunctionBodyContext ?: return

        // We don't care about public functions because other diagnostics are already reported on them
        if (inlineFunctionBodyContext.inlineFunEffectiveVisibility == EffectiveVisibility.Public) return

        if (c.callsOrAssignments.any { it is FirAnnotation }) return

        val symbol = expression.toResolvedCallableSymbol() ?: return
        if (symbol.effectiveVisibility is EffectiveVisibility.Local) return

        fun ConeKotlinType.reportIfLessVisible() {
            fullyExpandedType().forEachType { type ->
                val symbolEffectiveVisibility =
                    type.toClassLikeSymbol()
                        ?.let { it.publishedApiEffectiveVisibility ?: it.effectiveVisibility } ?: return@forEachType

                if (inlineFunctionBodyContext.isLessVisibleThanInlineFunction(symbolEffectiveVisibility)) {
                    reporter.reportOn(
                        expression.source, FirErrors.LESS_VISIBLE_TYPE_IN_INLINE_ACCESSED_SIGNATURE,
                        symbol,
                        symbolEffectiveVisibility,
                        type,
                        inlineFunctionBodyContext.inlineFunEffectiveVisibility,
                    )
                }
            }
        }

        symbol.contextParameterSymbols.forEach { it.resolvedReturnType.reportIfLessVisible() }
        symbol.receiverParameterSymbol?.resolvedType?.reportIfLessVisible()
        if (symbol is FirFunctionSymbol) {
            symbol.valueParameterSymbols.forEach { it.resolvedReturnType.reportIfLessVisible() }
        }
        symbol.typeParameterSymbols.forEach { typeParameterSymbol ->
            typeParameterSymbol.resolvedBounds.forEach { it.coneType.reportIfLessVisible() }
        }
        symbol.resolvedReturnType.reportIfLessVisible()

        fun FirRegularClassSymbol.reportIfLessVisible() {
            val containingClassLookupTag = getContainingClassLookupTag()
            val effectiveVisibility = visibility.toEffectiveVisibility(containingClassLookupTag, true)
            if (inlineFunctionBodyContext.isLessVisibleThanInlineFunction(effectiveVisibility) && publishedApiEffectiveVisibility == null) {
                reporter.reportOn(
                    expression.source,
                    FirErrors.LESS_VISIBLE_CONTAINING_CLASS_IN_INLINE,
                    symbol,
                    effectiveVisibility,
                    this,
                    inlineFunctionBodyContext.inlineFunEffectiveVisibility
                )
                // Stop recursion to prevent multiple errors
                return
            }
            containingClassLookupTag?.toRegularClassSymbol()?.reportIfLessVisible()
        }

        // We don't check the visibility of the declaration itself because we generate synthetic bridges if necessary
        // and it won't lead to runtime crashes.
        symbol.containingClassLookupTag()?.toRegularClassSymbol()?.reportIfLessVisible()
    }
}