// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl;

import ksp.com.intellij.openapi.Disposable;
import ksp.com.intellij.openapi.components.Service;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.util.SimpleModificationTracker;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.openapi.vfs.VirtualFileManager;
import ksp.com.intellij.openapi.vfs.newvfs.BulkFileListener;
import ksp.com.intellij.openapi.vfs.newvfs.events.VFileDeleteEvent;
import ksp.com.intellij.openapi.vfs.newvfs.events.VFileEvent;
import ksp.com.intellij.openapi.vfs.newvfs.events.VFilePropertyChangeEvent;
import ksp.com.intellij.psi.PsiJavaModule;
import ksp.com.intellij.util.messages.MessageBusConnection;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.List;

@Service(Service.Level.PROJECT)
public final class PsiJavaModuleModificationTracker extends SimpleModificationTracker implements Disposable {

  public static PsiJavaModuleModificationTracker getInstance(Project project) {
    return project.getService(PsiJavaModuleModificationTracker.class);
  }

  public PsiJavaModuleModificationTracker(Project project) {
    MessageBusConnection connect = project.getMessageBus().connect(this);
    connect.subscribe(VirtualFileManager.VFS_CHANGES, new BulkFileListener() {
      @Override
      public void after(@NotNull List<? extends @NotNull VFileEvent> events) {
        for (VFileEvent event : events) {
          VirtualFile file = event.getFile();
          if (file != null) {
            if (!file.isDirectory() && isModuleFile(file.getName()) ||
                event instanceof VFileDeleteEvent || //ensure inc when directory with MANIFEST.MF was deleted
                event instanceof VFilePropertyChangeEvent && //ensure inc when directory with MANIFEST.MF was renamed or manifest was renamed to a new name
                    VirtualFile.PROP_NAME.equals(((VFilePropertyChangeEvent)event).getPropertyName())) {
              incModificationCount();
              break;
            }
          }
        }
      }
    });
  }

  static boolean isModuleFile(String name) {
    return PsiJavaModule.MODULE_INFO_FILE.equals(name) || "MANIFEST.MF".equalsIgnoreCase(name);
  }

  @Override
  public void dispose() { }
}
