// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.lang;

import ksp.com.intellij.openapi.extensions.CustomLoadingExtensionPointBean;
import ksp.com.intellij.openapi.extensions.PluginDescriptor;
import ksp.com.intellij.openapi.extensions.RequiredElement;
import ksp.com.intellij.openapi.util.text.StringUtilRt;
import ksp.com.intellij.serviceContainer.NonInjectable;
import ksp.com.intellij.util.KeyedLazyInstance;
import ksp.com.intellij.util.xmlb.annotations.Attribute;
import ksp.org.jetbrains.annotations.ApiStatus;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.annotations.TestOnly;

/**
 * Base class for {@link Language}-bound extension points.
 */
public class LanguageExtensionPoint<T> extends CustomLoadingExtensionPointBean<T> implements KeyedLazyInstance<T> {
  // these must be public for scrambling compatibility
  /**
   * Language ID.
   *
   * @see Language#getID()
   */
  @Attribute("language")
  @RequiredElement(allowEmpty = true)
  public String language;

  @Attribute("implementationClass")
  public String implementationClass;

  /**
   * @deprecated You must pass plugin descriptor, use {@link LanguageExtensionPoint#LanguageExtensionPoint(String, Object)}
   */
  @Deprecated
  @ApiStatus.ScheduledForRemoval
  public LanguageExtensionPoint() {
  }

  @TestOnly
  @NonInjectable
  public LanguageExtensionPoint(@NotNull String language, @NotNull String implementationClass, @NotNull PluginDescriptor pluginDescriptor) {
    this.language = language;
    this.implementationClass = implementationClass;
    setPluginDescriptor(pluginDescriptor);
  }

  @TestOnly
  public LanguageExtensionPoint(@NotNull String language, @NotNull T instance) {
    super(instance);

    this.language = language;
    implementationClass = instance.getClass().getName();
  }

  @Override
  protected final @Nullable String getImplementationClassName() {
    return implementationClass;
  }

  @Override
  public @NotNull String getKey() {
    // empty string means any language
    return StringUtilRt.notNullize(language);
  }
}
