// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.model.psi;

import ksp.com.intellij.model.Symbol;
import ksp.com.intellij.openapi.util.TextRange;
import ksp.com.intellij.psi.PsiElement;
import ksp.org.jetbrains.annotations.NotNull;

/**
 * Symbol declaration in PSI tree,
 * which is a way to convey host element and range in host element to the platform, e.g., for highlighting and navigation.
 * <p/>
 * A symbol might be declared in several places, i.e. several declarations may declare the same symbol:
 * <pre>
 * SymbolDeclaration           d1   d2  dN
 *                               ↘  ↓  ↙
 * Symbol                           s
 * </pre>
 * <h4>Lifecycle</h4>
 * The PsiSymbolDeclaration instance is expected to stay valid within a single read action.
 * <p/>
 * <h4>Equality</h4>
 * There are no restrictions on whether implementations must provide {@link #equals}/{@link #hashCode}.
 *
 * @see PsiSymbolDeclarationProvider
 */
public interface PsiSymbolDeclaration {

  /**
   * @return underlying (declaring) element
   */
  @NotNull PsiElement getDeclaringElement();

  /**
   * @return range relative to {@link #getDeclaringElement() element} range,
   * which is considered a declaration, e.g. range of identifier in Java class
   */
  @NotNull TextRange getRangeInDeclaringElement();

  /**
   * @return range in the {@link PsiElement#getContainingFile containing file} of the {@link #getDeclaringElement() element}
   * which is considered a declaration
   * @see #getRangeInDeclaringElement()
   */
  default @NotNull TextRange getAbsoluteRange() {
    return getRangeInDeclaringElement().shiftRight(getDeclaringElement().getTextRange().getStartOffset());
  }

  @NotNull
  Symbol getSymbol();
}
