// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package ksp.com.intellij.psi.impl;

import ksp.com.intellij.lang.ASTFactory;
import ksp.com.intellij.lang.Commenter;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.lang.LanguageCommenters;
import ksp.com.intellij.openapi.fileTypes.LanguageFileType;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.psi.*;
import ksp.com.intellij.psi.impl.source.DummyHolderFactory;
import ksp.com.intellij.psi.impl.source.SourceTreeToPsiMap;
import ksp.com.intellij.psi.impl.source.tree.FileElement;
import ksp.com.intellij.psi.impl.source.tree.LeafElement;
import ksp.com.intellij.psi.impl.source.tree.TreeElement;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;


public final class PsiParserFacadeImpl implements PsiParserFacade {
  private final PsiManagerEx myManager;

  public PsiParserFacadeImpl(@NotNull Project project) {
    myManager = PsiManagerEx.getInstanceEx(project);
  }

  @Override
  public @NotNull PsiElement createWhiteSpaceFromText(@NotNull @NonNls String text) throws IncorrectOperationException {
    FileElement holderElement = DummyHolderFactory.createHolder(myManager, null).getTreeElement();
    LeafElement newElement = ASTFactory.leaf(TokenType.WHITE_SPACE, holderElement.getCharTable().intern(text));
    holderElement.rawAddChildren(newElement);
    GeneratedMarkerVisitor.markGenerated(newElement.getPsi());
    return newElement.getPsi();
  }

  @Override
  public @NotNull PsiComment createLineCommentFromText(@NotNull LanguageFileType fileType,
                                                       @NotNull String text) throws IncorrectOperationException {
    return createLineCommentFromText(fileType.getLanguage(), text);
  }

  @Override
  public @NotNull PsiComment createLineCommentFromText(@NotNull Language language,
                                                       @NotNull String text) throws IncorrectOperationException {
    Commenter commenter = LanguageCommenters.INSTANCE.forLanguage(language);
    assert commenter != null;
    String prefix = commenter.getLineCommentPrefix();
    if (prefix == null) {
      throw new IncorrectOperationException("No line comment prefix defined for language " + language.getID());
    }

    PsiFile aFile = createDummyFile(language, prefix + text);
    return findPsiCommentChild(aFile);
  }

  @Override
  public @NotNull PsiComment createBlockCommentFromText(@NotNull Language language,
                                                        @NotNull String text) throws IncorrectOperationException {
    Commenter commenter = LanguageCommenters.INSTANCE.forLanguage(language);
    assert commenter != null : language;
    String blockCommentPrefix = commenter.getBlockCommentPrefix();
    String blockCommentSuffix = commenter.getBlockCommentSuffix();
    assert blockCommentPrefix != null && blockCommentSuffix != null;

    PsiFile aFile = createDummyFile(language, blockCommentPrefix + text + blockCommentSuffix);
    return findPsiCommentChild(aFile);
  }

  @Override
  public @NotNull PsiComment createLineOrBlockCommentFromText(@NotNull Language language,
                                                              @NotNull String text) throws IncorrectOperationException {
    Commenter commenter = LanguageCommenters.INSTANCE.forLanguage(language);
    assert commenter != null : language;
    String prefix = commenter.getLineCommentPrefix();
    String blockCommentPrefix = commenter.getBlockCommentPrefix();
    String blockCommentSuffix = commenter.getBlockCommentSuffix();
    assert prefix != null || (blockCommentPrefix != null && blockCommentSuffix != null);

    PsiFile aFile = createDummyFile(language, prefix != null ? (prefix + text) : (blockCommentPrefix + text + blockCommentSuffix));
    return findPsiCommentChild(aFile);
  }

  private PsiComment findPsiCommentChild(PsiFile aFile) {
    PsiComment comment = PsiTreeUtil.findChildOfType(aFile, PsiComment.class);
    if (comment == null) {
      throw new IncorrectOperationException("Incorrect comment \"" + aFile.getText() + "\".");
    }

    DummyHolderFactory.createHolder(myManager, (TreeElement)SourceTreeToPsiMap.psiElementToTree(comment), null);
    return comment;
  }

  private PsiFile createDummyFile(Language language, String text) {
    return PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_Dummy_", language, text);
  }
}
