// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.impl.compiled;

import ksp.com.intellij.openapi.util.NotNullLazyValue;
import ksp.com.intellij.pom.Navigatable;
import ksp.com.intellij.psi.*;
import ksp.com.intellij.psi.impl.PsiImplUtil;
import ksp.com.intellij.psi.impl.java.stubs.PsiAnnotationStub;
import ksp.com.intellij.psi.impl.source.SourceTreeToPsiMap;
import ksp.com.intellij.psi.impl.source.tree.TreeElement;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

public final class ClsAnnotationImpl extends ClsRepositoryPsiElement<PsiAnnotationStub> implements PsiAnnotation, Navigatable {
  private final NotNullLazyValue<ClsJavaCodeReferenceElementImpl> myReferenceElement;
  private final NotNullLazyValue<ClsAnnotationParameterListImpl> myParameterList;

  public ClsAnnotationImpl(PsiAnnotationStub stub) {
    super(stub);
    myReferenceElement = NotNullLazyValue.atomicLazy(() -> {
      String annotationText = getStub().getText();
      int index = annotationText.indexOf('(');
      String refText = index > 0 ? annotationText.substring(1, index) : annotationText.substring(1);
      return new ClsJavaCodeReferenceElementImpl(this, refText);
    });
    myParameterList = NotNullLazyValue.atomicLazy(() -> {
      PsiNameValuePair[] attrs = getStub().getText().indexOf('(') > 0
                                 ? PsiTreeUtil.getRequiredChildOfType(getStub().getPsiElement(), PsiAnnotationParameterList.class)
                                   .getAttributes()
                                 : PsiNameValuePair.EMPTY_ARRAY;
      return new ClsAnnotationParameterListImpl(this, attrs);
    });
  }

  @Override
  public void appendMirrorText(int indentLevel, @NotNull StringBuilder buffer) {
    buffer.append('@').append(myReferenceElement.getValue().getCanonicalText());
    appendText(getParameterList(), indentLevel, buffer);
  }

  @Override
  protected void setMirror(@NotNull TreeElement element) throws InvalidMirrorException {
    setMirrorCheckingType(element, null);
    PsiAnnotation mirror = SourceTreeToPsiMap.treeToPsiNotNull(element);
    setMirror(getNameReferenceElement(), mirror.getNameReferenceElement());
    setMirror(getParameterList(), mirror.getParameterList());
  }

  @Override
  public PsiElement @NotNull [] getChildren() {
    return new PsiElement[]{myReferenceElement.getValue(), getParameterList()};
  }

  @Override
  public void accept(@NotNull PsiElementVisitor visitor) {
    if (visitor instanceof JavaElementVisitor) {
      ((JavaElementVisitor)visitor).visitAnnotation(this);
    }
    else {
      visitor.visitElement(this);
    }
  }

  @Override
  @NotNull
  public PsiAnnotationParameterList getParameterList() {
    return myParameterList.getValue();
  }

  @Override
  public boolean hasQualifiedName(@NotNull String qualifiedName) {
    String text = getStub().getText();
    // text is like '@<qualified_name>[(parameters)]'
    if (!text.startsWith(qualifiedName, 1)) return false;
    int endOfQualifiedName = qualifiedName.length() + 1;
    return text.length() == endOfQualifiedName || text.charAt(endOfQualifiedName) == '(';
  }

  @Override
  public @NotNull String getQualifiedName() {
    return myReferenceElement.getValue().getCanonicalText();
  }

  @Override
  public PsiJavaCodeReferenceElement getNameReferenceElement() {
    return myReferenceElement.getValue();
  }

  @Override
  public PsiAnnotationMemberValue findAttributeValue(String attributeName) {
    return PsiImplUtil.findAttributeValue(this, attributeName);
  }

  @Override
  @Nullable
  public PsiAnnotationMemberValue findDeclaredAttributeValue(@NonNls final String attributeName) {
    return PsiImplUtil.findDeclaredAttributeValue(this, attributeName);
  }

  @Override
  public <T extends PsiAnnotationMemberValue> T setDeclaredAttributeValue(@NonNls String attributeName, T value) {
    throw cannotModifyException(this);
  }

  @Override
  public String getText() {
    final StringBuilder buffer = new StringBuilder();
    appendMirrorText(0, buffer);
    return buffer.toString();
  }

  @Override
  public PsiAnnotationOwner getOwner() {
    return (PsiAnnotationOwner)getParent();//todo
  }
}
