// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.impl.java.stubs.impl;

import ksp.com.intellij.psi.PsiParameter;
import ksp.com.intellij.psi.impl.cache.TypeInfo;
import ksp.com.intellij.psi.impl.java.stubs.JavaStubElementTypes;
import ksp.com.intellij.psi.impl.java.stubs.PsiModifierListStub;
import ksp.com.intellij.psi.impl.java.stubs.PsiParameterStub;
import ksp.com.intellij.psi.stubs.StubBase;
import ksp.com.intellij.psi.stubs.StubElement;
import ksp.com.intellij.util.BitUtil;
import ksp.org.jetbrains.annotations.NotNull;

public class PsiParameterStubImpl extends StubBase<PsiParameter> implements PsiParameterStub {
  private final static byte ELLIPSIS = 0x01;
  private final static byte GENERATED_NAME = 0x02;

  private static byte packFlags(boolean isEllipsis, boolean generatedName) {
    byte flags = 0;
    flags = BitUtil.set(flags, ELLIPSIS, isEllipsis);
    flags = BitUtil.set(flags, GENERATED_NAME, generatedName);
    return flags;
  }

  private String myName;
  private final TypeInfo myType;
  private byte myFlags;

  public PsiParameterStubImpl(StubElement parent, @NotNull String name, @NotNull TypeInfo type, boolean ellipsis, boolean generatedName) {
    this(parent, name, type, packFlags(ellipsis, generatedName));
  }

  public PsiParameterStubImpl(StubElement parent, @NotNull String name, @NotNull TypeInfo type, byte flags) {
    super(parent, JavaStubElementTypes.PARAMETER);
    myName = name;
    myType = type;
    myFlags = flags;
  }

  @Override
  public boolean isParameterTypeEllipsis() {
    return BitUtil.isSet(myFlags, ELLIPSIS);
  }

  @Override
  @NotNull
  public TypeInfo getType() {
    return myType;
  }

  @Override
  public PsiModifierListStub getModList() {
    for (StubElement child : getChildrenStubs()) {
      if (child instanceof PsiModifierListStub) {
        return (PsiModifierListStub)child;
      }
    }
    return null;
  }

  @NotNull
  @Override
  public String getName() {
    return myName;
  }

  public void setName(String name) {
    myName = name;
    myFlags = BitUtil.set(myFlags, GENERATED_NAME, false);
  }

  public boolean isAutoGeneratedName() {
    return BitUtil.isSet(myFlags, GENERATED_NAME);
  }

  public byte getFlags() {
    return myFlags;
  }

  @Override
  public String toString() {
    return "PsiParameterStub[" + myName + ':' + myType + ']';
  }
}