// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl.source.javadoc;

import ksp.com.intellij.openapi.util.TextRange;
import ksp.com.intellij.psi.AbstractElementManipulator;
import ksp.com.intellij.psi.JavaPsiFacade;
import ksp.com.intellij.psi.PsiElementFactory;
import ksp.com.intellij.psi.codeStyle.JavaFileCodeStyleFacade;
import ksp.com.intellij.psi.javadoc.PsiDocComment;
import ksp.com.intellij.psi.javadoc.PsiSnippetDocTag;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.org.jetbrains.annotations.Contract;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.List;

public final class SnippetDocTagManipulator extends AbstractElementManipulator<PsiSnippetDocTagImpl> {

  @Override
  public PsiSnippetDocTagImpl handleContentChange(@NotNull PsiSnippetDocTagImpl element,
                                                  @NotNull TextRange range,
                                                  String newContent) throws IncorrectOperationException {
    final PsiElementFactory factory = JavaPsiFacade.getElementFactory(element.getProject());

    final JavaFileCodeStyleFacade codeStyleFacade = JavaFileCodeStyleFacade.forContext(element.getContainingFile());
    final String newSnippetTagContent = codeStyleFacade.isJavaDocLeadingAsterisksEnabled()
                                        ? prependAbsentAsterisks(newContent)
                                        : newContent;

    final PsiDocComment text = factory.createDocCommentFromText("/**\n" + newSnippetTagContent + "\n*/");
    final PsiSnippetDocTag snippet = PsiTreeUtil.findChildOfType(text, PsiSnippetDocTag.class);
    if (snippet == null) {
      return element;
    }
    return (PsiSnippetDocTagImpl)element.replace(snippet);
  }

  @Contract(pure = true)
  private static @NotNull String prependAbsentAsterisks(@NotNull String input) {
    return input.replaceAll("(\\n\\s*)([^*\\s])", "$1 * $2");
  }

  @Override
  public @NotNull TextRange getRangeInElement(@NotNull PsiSnippetDocTagImpl element) {
    final List<TextRange> ranges = element.getContentRanges();
    if (ranges.isEmpty()) return TextRange.EMPTY_RANGE;
    final int startOffset = ranges.get(0).getStartOffset();
    final int endOffset = ContainerUtil.getLastItem(ranges).getEndOffset();
    return TextRange.create(startOffset, endOffset);
  }
}
