// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl.source.resolve.reference;

import ksp.com.intellij.openapi.project.IndexNotReadyException;
import ksp.com.intellij.openapi.util.text.StringUtil;
import ksp.com.intellij.patterns.ElementPattern;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiReferenceProvider;
import ksp.com.intellij.psi.PsiReferenceService;
import ksp.com.intellij.util.ProcessingContext;
import ksp.com.intellij.util.SharedProcessingContext;
import ksp.com.intellij.util.SmartList;
import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author maxim
 */
public abstract class NamedObjectProviderBinding implements ProviderBinding {
  private final Map<String, List<@NotNull ProviderInfo<ElementPattern<?>>>> myNamesToProvidersMap = new HashMap<>(5);
  private final Map<String, List<@NotNull ProviderInfo<ElementPattern<?>>>> myNamesToProvidersMapInsensitive = new HashMap<>(5);

  public void registerProvider(@NonNls String @NotNull [] names,
                               @NotNull ElementPattern<?> filter,
                               boolean caseSensitive,
                               @NotNull PsiReferenceProvider provider,
                               double priority) {
    Map<String, List<ProviderInfo<ElementPattern<?>>>> map = caseSensitive ? myNamesToProvidersMap : myNamesToProvidersMapInsensitive;

    for (String attributeName : names) {
      String key = caseSensitive ? attributeName : StringUtil.toLowerCase(attributeName);
      List<ProviderInfo<ElementPattern<?>>> psiReferenceProviders = map.get(key);

      if (psiReferenceProviders == null) {
        map.put(key, psiReferenceProviders = new SmartList<>());
      }

      psiReferenceProviders.add(new ProviderInfo<>(provider, filter, priority));
    }
  }

  @Override
  public void addAcceptableReferenceProviders(@NotNull PsiElement position,
                                              @NotNull List<? super ProviderInfo<ProcessingContext>> list,
                                              @NotNull PsiReferenceService.Hints hints) {
    String name = getName(position);
    if (name != null) {
      addMatchingProviders(position, ContainerUtil.notNullize(myNamesToProvidersMap.get(name)), list, hints);
      addMatchingProviders(position, ContainerUtil.notNullize(myNamesToProvidersMapInsensitive.get(StringUtil.toLowerCase(name))), list, hints);
    }
  }

  @Override
  public void unregisterProvider(@NotNull PsiReferenceProvider provider) {
    for (List<ProviderInfo<ElementPattern<?>>> list : myNamesToProvidersMap.values()) {
      list.removeIf(trinity -> trinity.provider.equals(provider));
    }
    for (List<ProviderInfo<ElementPattern<?>>> list : myNamesToProvidersMapInsensitive.values()) {
      list.removeIf(trinity -> trinity.provider.equals(provider));
    }
  }

  boolean isEmpty() {
    return myNamesToProvidersMap.isEmpty() && myNamesToProvidersMapInsensitive.isEmpty();
  }

  protected abstract @Nullable String getName(@NotNull PsiElement position);

  static void addMatchingProviders(@NotNull PsiElement position,
                                   @NotNull List<? extends @NotNull ProviderInfo<ElementPattern<?>>> providerList,
                                   @NotNull Collection<? super ProviderInfo<ProcessingContext>> output,
                                   @NotNull PsiReferenceService.Hints hints) {
    SharedProcessingContext sharedProcessingContext = new SharedProcessingContext();

    for (ProviderInfo<ElementPattern<?>> info : providerList) {
      if (hints != PsiReferenceService.Hints.NO_HINTS && !info.provider.acceptsHints(position, hints)) {
        continue;
      }

      ProcessingContext context = new ProcessingContext(sharedProcessingContext);
      boolean suitable = false;
      try {
        suitable = info.processingContext.accepts(position, context);
      }
      catch (IndexNotReadyException ignored) {
      }
      if (suitable) {
        output.add(new ProviderInfo<>(info.provider, context, info.priority));
      }
    }
  }
}
