// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util;

import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.util.Iconable;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.psi.PsiManager;
import ksp.com.intellij.testFramework.LightVirtualFile;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.Objects;

/**
 * @author Konstantin Bulenkov
 */
final class FileIconKey {
  private final VirtualFile myFile;
  private final Project myProject;
  @Iconable.IconFlags private final int myFlags;
  private final @Nullable Language myInitialLanguage;
  private final long myStamp;

  FileIconKey(@NotNull VirtualFile file, @Nullable Project project, @Iconable.IconFlags int flags) {
    myFile = file;
    myProject = project;
    myFlags = flags;
    myInitialLanguage = myFile instanceof LightVirtualFile ? ((LightVirtualFile)myFile).getLanguage() : null;
    myStamp = project == null ? 0 : PsiManager.getInstance(project).getModificationTracker().getModificationCount();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof FileIconKey)) return false;

    FileIconKey that = (FileIconKey)o;
    if (myFlags != that.myFlags) return false;
    if (myStamp != that.myStamp) return false;
    if (!myFile.equals(that.myFile)) return false;
    if (!Objects.equals(myProject, that.myProject)) return false;

    if (!Objects.equals(myInitialLanguage, that.myInitialLanguage)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return Objects.hash(myFile, myProject, myFlags, myStamp);
  }

  VirtualFile getFile() {
    return myFile;
  }

  Project getProject() {
    return myProject;
  }

  @Iconable.IconFlags int getFlags() {
    return myFlags;
  }
}