// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util

import ksp.com.intellij.DynamicBundle
import ksp.com.intellij.DynamicBundle.getLocale
import ksp.com.intellij.openapi.util.io.FileUtil
import ksp.org.jetbrains.annotations.ApiStatus
import java.io.InputStream
import java.nio.file.Path
import java.util.*
import kotlin.io.path.Path
import kotlin.io.path.extension
import kotlin.io.path.nameWithoutExtension
import kotlin.io.path.pathString

@ApiStatus.Internal
object LocalizationUtil {
  @Volatile
  var isL10nPluginInitialized: Boolean = false
  private const val LOCALIZATION_FOLDER_NAME = "localization"
    private fun getPluginClassLoader(): ClassLoader? = DynamicBundle.findLanguageBundle()?.pluginDescriptor?.pluginClassLoader
    private fun Path.convertToLocalizationFolderUsage(locale: Locale, withRegion: Boolean): Path {
      var result = Path(LOCALIZATION_FOLDER_NAME).resolve(locale.language)
      if (withRegion && locale.country.isNotEmpty()) {
        result = result.resolve(locale.country)
      }
      result = result.resolve(this)
      return result
    }

    private fun Path.convertPathToLocaleSuffixUsage(locale: Locale?, withRegion: Boolean): Path {
      if (locale == null) return this
      val fileName = StringBuilder(this.nameWithoutExtension)
      val extension = this.extension
      val foldersPath = this.parent ?: Path("")
      val language = locale.language
      if (!language.isEmpty()) {
        fileName.append('_').append(language)
        val country = locale.country
        if (country.isNotEmpty() && withRegion) {
          fileName.append('_').append(country)
        }
      }
      if (extension.isNotEmpty()) {
        fileName.append(".").append(extension)
      }
      val result = foldersPath.resolve(fileName.toString())
      return result
    }

    @JvmOverloads
    fun getResourceAsStream(defaultLoader: ClassLoader?, path: Path,  specialLocale: Locale? = null): InputStream? {
      val locale = specialLocale ?: getLocale()
      val localizedPaths = getLocalizedPaths(path, locale)
      for (localizedPath in localizedPaths) {
        val pathString = FileUtil.toSystemIndependentName(localizedPath.pathString)
        getPluginClassLoader()?.getResourceAsStream(pathString)?.let { return it }
        defaultLoader?.getResourceAsStream(pathString)?.let { return it }
      }
      return null
    }

    @JvmOverloads
    fun getLocalizedPaths(path: Path, specialLocale: Locale? = null): List<Path> {
      val locale = specialLocale ?: getLocale()
      return listOf(
        //localizations/zh/CN/inspectionDescriptions/name.html
        path.convertToLocalizationFolderUsage(locale, true),

        //inspectionDescriptions/name_zh_CN.html
        path.convertPathToLocaleSuffixUsage(locale, true),

        //localizations/zh/inspectionDescriptions/name.html
        path.convertToLocalizationFolderUsage(locale, false),

        //inspectionDescriptions/name_zh.html
        path.convertPathToLocaleSuffixUsage(locale, false),

        //inspectionDescriptions/name.html
        path
      )
    }

    @JvmOverloads
    fun getFolderLocalizedPaths(path: Path, specialLocale: Locale? = null): List<Path> {
      val locale = specialLocale ?: getLocale()
      return listOf(
      //localizations/zh/CN/inspectionDescriptions/name.html
      path.convertToLocalizationFolderUsage(locale, true),

      //localizations/zh/inspectionDescriptions/name.html
      path.convertToLocalizationFolderUsage(locale, false))
    }
}