// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util.containers;

import ksp.it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import ksp.it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.function.Predicate;

public final class BiDirectionalEnumerator<T> extends Enumerator<T> {
  private final @NotNull Int2ObjectMap<T> myIntToObjectMap;

  public BiDirectionalEnumerator(int expectNumber) {
    super(expectNumber);

    myIntToObjectMap = new Int2ObjectOpenHashMap<>(expectNumber);
  }

  @Override
  public int enumerateImpl(T object) {
    int index = super.enumerateImpl(object);
    myIntToObjectMap.put(Math.max(index, -index), object);
    return index;
  }

  @Override
  public void clear() {
    super.clear();
    myIntToObjectMap.clear();
  }

  public @NotNull T getValue(int index) {
    T value = myIntToObjectMap.get(index);
    if (value == null) {
      throw new RuntimeException("Can not find value by index " + index);
    }
    return value;
  }

  public void forEachValue(@NotNull Predicate<? super T> processor) {
    for (T value : myIntToObjectMap.values()) {
      if (!processor.test(value)) {
        break;
      }
    }
  }
}
