/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.components

import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.KaContextParameterApi
import ksp.org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.analysis.api.resolve.extensions.KaResolveExtension
import ksp.org.jetbrains.kotlin.analysis.api.scopes.KaScope
import ksp.org.jetbrains.kotlin.psi.KtElement

@KaExperimentalApi
@SubclassOptInRequired(KaImplementationDetail::class)
public interface KaResolveExtensionInfoProvider : KaSessionComponent {
    /**
     * Returns the [KaScope] containing top-level callable declarations generated by [KaResolveExtension]s.
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val resolveExtensionScopeWithTopLevelDeclarations: KaScope

    /**
     * Whether the given [VirtualFile] is provided by a [KaResolveExtension].
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val VirtualFile.isResolveExtensionFile: Boolean

    /**
     * Whether the given [KtElement] is provided by a [KaResolveExtension].
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val KtElement.isFromResolveExtension: Boolean

    /**
     * A list of [PsiElement]s which are used as navigation targets in place of this [KtElement] provided by a [KaResolveExtension].
     *
     * These [PsiElement]s will typically be the source item(s) that caused the given [KtElement] to be generated by the
     * [KaResolveExtension]. For example, for a [KtElement] generated by a resource compiler, the navigation elements will typically be the
     * [PsiElement]s of the resource items in the corresponding resource file.
     *
     * @see KaResolveExtension
     * @see org.jetbrains.kotlin.analysis.api.resolve.extensions.KaResolveExtensionNavigationTargetsProvider
     */
    @KaExperimentalApi
    public val KtElement.resolveExtensionNavigationElements: Collection<PsiElement>
}

/**
 * @see KaResolveExtensionInfoProvider.resolveExtensionScopeWithTopLevelDeclarations
 */
@KaExperimentalApi
@KaContextParameterApi
context(context: KaResolveExtensionInfoProvider)
public val resolveExtensionScopeWithTopLevelDeclarations: KaScope
    get() = with(context) { resolveExtensionScopeWithTopLevelDeclarations }

/**
 * @see KaResolveExtensionInfoProvider.isResolveExtensionFile
 */
@KaExperimentalApi
@KaContextParameterApi
context(context: KaResolveExtensionInfoProvider)
public val VirtualFile.isResolveExtensionFile: Boolean
    get() = with(context) { isResolveExtensionFile }

/**
 * @see KaResolveExtensionInfoProvider.isFromResolveExtension
 */
@KaExperimentalApi
@KaContextParameterApi
context(context: KaResolveExtensionInfoProvider)
public val KtElement.isFromResolveExtension: Boolean
    get() = with(context) { isFromResolveExtension }

/**
 * @see KaResolveExtensionInfoProvider.resolveExtensionNavigationElements
 */
@KaExperimentalApi
@KaContextParameterApi
context(context: KaResolveExtensionInfoProvider)
public val KtElement.resolveExtensionNavigationElements: Collection<PsiElement>
    get() = with(context) { resolveExtensionNavigationElements }
