/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.components

import ksp.org.jetbrains.kotlin.analysis.api.KaContextParameterApi
import ksp.org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.analysis.api.signatures.KaCallableSignature
import ksp.org.jetbrains.kotlin.analysis.api.signatures.KaFunctionSignature
import ksp.org.jetbrains.kotlin.analysis.api.signatures.KaVariableSignature
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaCallableSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaFunctionSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaVariableSymbol
import ksp.org.jetbrains.kotlin.analysis.api.types.KaSubstitutor

@KaExperimentalApi
@SubclassOptInRequired(KaImplementationDetail::class)
public interface KaSignatureSubstitutor : KaSessionComponent {
    /**
     * Applies a [substitutor] to the given symbol and returns a [KaCallableSignature] with substituted types.
     *
     * @see KaSubstitutor.substitute
     */
    @KaExperimentalApi
    public fun <S : KaCallableSymbol> S.substitute(substitutor: KaSubstitutor): KaCallableSignature<S>

    /**
     * Applies a [substitutor] to the given symbol and returns a [KaFunctionSignature] with substituted types.
     *
     * @see KaSubstitutor.substitute
     */
    @KaExperimentalApi
    public fun <S : KaFunctionSymbol> S.substitute(substitutor: KaSubstitutor): KaFunctionSignature<S>

    /**
     * Applies a [substitutor] to the given symbol and returns a [KaVariableSignature] with substituted types.
     *
     * @see KaSubstitutor.substitute
     */
    @KaExperimentalApi
    public fun <S : KaVariableSymbol> S.substitute(substitutor: KaSubstitutor): KaVariableSignature<S>

    /**
     * Creates a new [KaCallableSignature] for the given symbol and leaves all types unsubstituted.
     */
    @KaExperimentalApi
    public fun <S : KaCallableSymbol> S.asSignature(): KaCallableSignature<S>

    /**
     * Creates a new [KaFunctionSignature] for the given symbol and leaves all types unsubstituted.
     */
    @KaExperimentalApi
    public fun <S : KaFunctionSymbol> S.asSignature(): KaFunctionSignature<S>

    /**
     * Creates a new [KaVariableSignature] for the given symbol and leaves all types unsubstituted.
     */
    @KaExperimentalApi
    public fun <S : KaVariableSymbol> S.asSignature(): KaVariableSignature<S>
}

/**
 * @see KaSignatureSubstitutor.substitute
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaCallableSymbol> S.substitute(substitutor: KaSubstitutor): KaCallableSignature<S> {
    return with(context) { substitute(substitutor) }
}

/**
 * @see KaSignatureSubstitutor.substitute
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaFunctionSymbol> S.substitute(substitutor: KaSubstitutor): KaFunctionSignature<S> {
    return with(context) { substitute(substitutor) }
}

/**
 * @see KaSignatureSubstitutor.substitute
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaVariableSymbol> S.substitute(substitutor: KaSubstitutor): KaVariableSignature<S> {
    return with(context) { substitute(substitutor) }
}

/**
 * @see KaSignatureSubstitutor.asSignature
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaCallableSymbol> S.asSignature(): KaCallableSignature<S> {
    return with(context) { asSignature() }
}

/**
 * @see KaSignatureSubstitutor.asSignature
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaFunctionSymbol> S.asSignature(): KaFunctionSignature<S> {
    return with(context) { asSignature() }
}

/**
 * @see KaSignatureSubstitutor.asSignature
 */
@KaContextParameterApi
@KaExperimentalApi
context(context: KaSignatureSubstitutor)
public fun <S : KaVariableSymbol> S.asSignature(): KaVariableSignature<S> {
    return with(context) { asSignature() }
}
