/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.types

import ksp.org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.analysis.api.KaNonPublicApi
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaSymbolByFirBuilder
import ksp.org.jetbrains.kotlin.analysis.api.fir.annotations.KaFirAnnotationListForType
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.createPointer
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.KaLifetimeToken
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.types.KaErrorType
import ksp.org.jetbrains.kotlin.analysis.api.types.KaTypePointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaUsualClassType
import ksp.org.jetbrains.kotlin.analysis.utils.errors.requireIsInstance
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeCannotInferTypeParameterType
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeTypeVariableTypeIsNotInferred
import ksp.org.jetbrains.kotlin.fir.types.ConeErrorType
import ksp.org.jetbrains.kotlin.fir.types.renderForDebugging

internal class KaFirErrorType(
    override val coneType: ConeErrorType,
    private val builder: KaSymbolByFirBuilder,
) : KaErrorType, KaFirType {

    override val token: KaLifetimeToken get() = builder.token

    @Deprecated(
        "Use `isMarkedNullable`, `isNullable` or `hasFlexibleNullability` instead. See KDocs for the migration guide",
        replaceWith = ReplaceWith("this.isMarkedNullable")
    )
    @Suppress("Deprecation")
    override val nullability: org.jetbrains.kotlin.analysis.api.types.KaTypeNullability
        get() = withValidityAssertion {
            coneType.nullable?.let(org.jetbrains.kotlin.analysis.api.types.KaTypeNullability::create)
                ?: org.jetbrains.kotlin.analysis.api.types.KaTypeNullability.UNKNOWN
        }

    @KaNonPublicApi
    override val errorMessage: String
        get() = withValidityAssertion { coneType.diagnostic.reason }

    @KaNonPublicApi
    override val presentableText: String?
        get() = withValidityAssertion {
            when (val diagnostic = coneType.diagnostic) {
                is ConeCannotInferTypeParameterType -> diagnostic.typeParameter.name.asString()
                is ConeTypeVariableTypeIsNotInferred -> diagnostic.typeVariableType.typeConstructor.debugName
                else -> null
            }
        }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            KaFirAnnotationListForType.create(coneType, builder)
        }

    override val abbreviation: KaUsualClassType?
        get() = withValidityAssertion { null }

    override fun equals(other: Any?) = typeEquals(other)
    override fun hashCode() = typeHashcode()
    override fun toString() = coneType.renderForDebugging()

    @KaExperimentalApi
    override fun createPointer(): KaTypePointer<KaErrorType> = withValidityAssertion {
        return KaFirErrorTypePointer(coneType, builder)
    }
}

private class KaFirErrorTypePointer(
    coneType: ConeErrorType,
    builder: KaSymbolByFirBuilder,
) : KaTypePointer<KaErrorType> {
    private val coneTypePointer = coneType.createPointer(builder)

    @KaImplementationDetail
    override fun restore(session: KaSession): KaErrorType? = session.withValidityAssertion {
        requireIsInstance<KaFirSession>(session)

        val coneType = coneTypePointer.restore(session) ?: return null
        return KaFirErrorType(coneType, session.firSymbolBuilder)
    }
}