/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.renderer.declarations

import ksp.org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotated
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaDeclarationSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaEnumEntrySymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaTypeParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType

@KaExperimentalApi
public interface KaRendererCodeStyle {
    public fun getIndentSize(analysisSession: KaSession): Int

    public fun getSeparatorAfterContextReceivers(analysisSession: KaSession): String

    public fun getSeparatorBetweenAnnotationAndOwner(analysisSession: KaSession, symbol: KaAnnotated): String

    public fun getSeparatorBetweenAnnotations(analysisSession: KaSession, symbol: KaAnnotated): String

    public fun getSeparatorBetweenModifiers(analysisSession: KaSession): String

    public fun getSeparatorBetweenMembers(
        analysisSession: KaSession,
        first: KaDeclarationSymbol,
        second: KaDeclarationSymbol,
    ): String
}

@KaExperimentalApi
public object KaRecommendedRendererCodeStyle : KaRendererCodeStyle {
    override fun getIndentSize(analysisSession: KaSession): Int = 4

    override fun getSeparatorAfterContextReceivers(analysisSession: KaSession): String = "\n"

    override fun getSeparatorBetweenAnnotationAndOwner(analysisSession: KaSession, symbol: KaAnnotated): String = when (symbol) {
        is KaType -> " "
        is KaTypeParameterSymbol -> " "
        is KaParameterSymbol -> " "
        else -> "\n"
    }

    override fun getSeparatorBetweenAnnotations(analysisSession: KaSession, symbol: KaAnnotated): String = when (symbol) {
        is KaType -> " "
        is KaTypeParameterSymbol -> " "
        is KaParameterSymbol -> " "
        else -> "\n"
    }

    override fun getSeparatorBetweenModifiers(analysisSession: KaSession): String = " "

    override fun getSeparatorBetweenMembers(analysisSession: KaSession, first: KaDeclarationSymbol, second: KaDeclarationSymbol): String {
        return when {
            first is KaEnumEntrySymbol && second is KaEnumEntrySymbol -> ",\n"
            first is KaEnumEntrySymbol && second !is KaEnumEntrySymbol -> ";\n\n"
            else -> "\n\n"
        }
    }
}
