/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.combined

import ksp.com.github.benmanes.caffeine.cache.Cache
import ksp.com.github.benmanes.caffeine.cache.Caffeine
import ksp.com.intellij.openapi.project.Project
import ksp.org.jetbrains.kotlin.analysis.api.platform.declarations.KotlinDeclarationProvider
import ksp.org.jetbrains.kotlin.analysis.api.platform.declarations.mergeDeclarationProviders
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.KotlinPackageProvider
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.mergePackageProviders
import ksp.org.jetbrains.kotlin.analysis.api.platform.caches.NullableCaffeineCache
import ksp.org.jetbrains.kotlin.analysis.api.platform.caches.getOrPut
import ksp.org.jetbrains.kotlin.analysis.api.platform.caches.withStatsCounter
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLKotlinSymbolProvider
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions.LLFirSession
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.statistics.LLStatisticsService
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLModuleSpecificSymbolProviderAccess
import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirCompositeCachedSymbolNamesProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolNamesProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProviderInternals
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.name.CallableId
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.psi.KtCallableDeclaration
import java.time.Duration

/**
 * [LLCombinedKotlinSymbolProvider] combines multiple [LLKotlinSymbolProvider]s with the following advantages:
 *
 * - The combined symbol provider can combine the "names in package" sets built by individual providers. The name set can then be checked
 *   once instead of for each subordinate symbol provider. Because Kotlin symbol providers are ordered first in
 *   [LLDependenciesSymbolProvider][org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLDependenciesSymbolProvider],
 *   this check is especially fruitful.
 * - For a given class or callable ID, indices can be accessed once to get relevant PSI elements. Then the correct symbol provider(s) to
 *   call can be found out via the PSI element's [KaModule][org.jetbrains.kotlin.analysis.api.projectStructure.KaModule]s. This avoids the
 *   need to call every single subordinate symbol provider.
 * - A small Caffeine cache can avoid most index accesses for classes, because many names are requested multiple times, with a minor memory
 *   footprint.
 * - Caffeine caches for functions and properties use time-based eviction, which allows them to scale up in short bursts when many callables
 *   are requested.
 *
 * @param declarationProvider The declaration provider must have a scope which combines the scopes of the individual [providers].
 *
 * @param packageProviderForKotlinPackages This package provider should be combined from all [providers] which allow `kotlin` packages (see
 *  [LLKotlinSymbolProvider.allowKotlinPackage]). It may be `null` if no such provider exists. See [hasPackage] for a use case.
 */
internal class LLCombinedKotlinSymbolProvider private constructor(
    session: FirSession,
    project: Project,
    providers: List<LLKotlinSymbolProvider>,
    private val declarationProvider: KotlinDeclarationProvider,
    private val packageProvider: KotlinPackageProvider,
    private val packageProviderForKotlinPackages: KotlinPackageProvider?,
) : LLSelectingCombinedSymbolProvider<LLKotlinSymbolProvider>(session, project, providers) {
    override val symbolNamesProvider: FirSymbolNamesProvider = FirCompositeCachedSymbolNamesProvider.fromSymbolProviders(session, providers)

    private val classifierCache = NullableCaffeineCache<ClassId, FirClassLikeSymbol<*>> {
        it
            .maximumSize(500)
            .withStatsCounter(LLStatisticsService.getInstance(project)?.symbolProviders?.combinedSymbolProviderClassCacheStatsCounter)
    }

    private val functionCache =
        Caffeine.newBuilder()
            .expireAfterAccess(Duration.ofSeconds(5))
            .withStatsCounter(LLStatisticsService.getInstance(project)?.symbolProviders?.combinedSymbolProviderCallableCacheStatsCounter)
            .build<CallableId, List<FirNamedFunctionSymbol>>()

    private val propertyCache =
        Caffeine.newBuilder()
            .expireAfterAccess(Duration.ofSeconds(5))
            .withStatsCounter(LLStatisticsService.getInstance(project)?.symbolProviders?.combinedSymbolProviderCallableCacheStatsCounter)
            .build<CallableId, List<FirPropertySymbol>>()

    override fun getClassLikeSymbolByClassId(classId: ClassId): FirClassLikeSymbol<*>? {
        if (!symbolNamesProvider.mayHaveTopLevelClassifier(classId)) return null

        return classifierCache.getOrPut(classId) { computeClassLikeSymbolByClassId(it) }
    }

    private fun computeClassLikeSymbolByClassId(classId: ClassId): FirClassLikeSymbol<*>? {
        val candidates = declarationProvider.getAllClassesByClassId(classId) + declarationProvider.getAllTypeAliasesByClassId(classId)
        val (ktClass, provider) = selectFirstElementInClasspathOrder(candidates) { it } ?: return null

        // We've picked the symbol provider via the `ktClass`, so `ktClass` must be contained in the symbol provider's module.
        @OptIn(LLModuleSpecificSymbolProviderAccess::class)
        return provider.getClassLikeSymbolByClassId(classId, ktClass)
    }

    @FirSymbolProviderInternals
    override fun getTopLevelCallableSymbolsTo(destination: MutableList<FirCallableSymbol<*>>, packageFqName: FqName, name: Name) {
        if (!symbolNamesProvider.mayHaveTopLevelCallable(packageFqName, name)) return

        val callableId = CallableId(packageFqName, name)

        // Callables are provided very rarely (compared to functions/properties individually), so it's acceptable to hit caches and indices
        // twice here.
        destination.addAll(getTopLevelFunctionSymbolsFromCache(callableId))
        destination.addAll(getTopLevelPropertySymbolsFromCache(callableId))
    }

    @FirSymbolProviderInternals
    override fun getTopLevelFunctionSymbolsTo(destination: MutableList<FirNamedFunctionSymbol>, packageFqName: FqName, name: Name) {
        if (!symbolNamesProvider.mayHaveTopLevelCallable(packageFqName, name)) return

        destination.addAll(getTopLevelFunctionSymbolsFromCache(CallableId(packageFqName, name)))
    }

    @FirSymbolProviderInternals
    override fun getTopLevelPropertySymbolsTo(destination: MutableList<FirPropertySymbol>, packageFqName: FqName, name: Name) {
        if (!symbolNamesProvider.mayHaveTopLevelCallable(packageFqName, name)) return

        destination.addAll(getTopLevelPropertySymbolsFromCache(CallableId(packageFqName, name)))
    }

    @OptIn(FirSymbolProviderInternals::class)
    private fun getTopLevelFunctionSymbolsFromCache(callableId: CallableId): List<FirNamedFunctionSymbol> =
        getCallablesFromCache(
            callableId,
            functionCache,
            declarationProvider::getTopLevelFunctions,
        ) { destination, callableId, functions ->
            getTopLevelFunctionSymbolsTo(destination, callableId, functions)
        }

    @OptIn(FirSymbolProviderInternals::class)
    private fun getTopLevelPropertySymbolsFromCache(callableId: CallableId): List<FirPropertySymbol> =
        getCallablesFromCache(
            callableId,
            propertyCache,
            declarationProvider::getTopLevelProperties,
        ) { destination, callableId, properties ->
            getTopLevelPropertySymbolsTo(destination, callableId, properties)
        }

    /**
     * Retrieves all callables of type [S] from the given [cache] or loads them with [getCallables] and [provide].
     *
     * We cannot use [KotlinDeclarationProvider.getTopLevelCallableFiles] like [LLKotlinSourceSymbolProvider][org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLKotlinSourceSymbolProvider]
     * for optimization because this approach only works for sources. Stub-based library symbol providers shouldn't access callables from
     * [KtFile][org.jetbrains.kotlin.psi.KtFile]s.
     */
    private inline fun <A : KtCallableDeclaration, S : FirCallableSymbol<*>> getCallablesFromCache(
        callableId: CallableId,
        cache: Cache<CallableId, List<S>>,
        crossinline getCallables: (CallableId) -> Collection<A>,
        crossinline provide: LLKotlinSymbolProvider.(MutableList<S>, CallableId, Collection<A>) -> Unit,
    ): List<S> =
        cache.getOrPut(callableId) {
            buildList {
                getCallables(callableId)
                    .groupBy { getModule(it) }
                    .forEach { (module, callables) ->
                        // If `module` cannot be found in the map, `callables` cannot be processed by any of the available providers,
                        // because none of them belong to the correct module. We can skip in that case because iterating through all
                        // providers wouldn't lead to any results for `callables`.
                        val provider = getProviderByModule(module) ?: return@forEach
                        provider.provide(this, callableId, callables)
                    }
            }
        }

    override fun hasPackage(fqName: FqName): Boolean {
        val hasPackage = if (fqName.startsWith(StandardNames.BUILT_INS_PACKAGE_NAME)) {
            // If a package is a `kotlin` package, `packageProvider` might find it via the scope of an individual symbol provider that
            // disallows `kotlin` packages. Hence, the combined `getPackage` would erroneously find a package it shouldn't be able to find,
            // because calling that individual symbol provider directly would result in `null` (as it disallows `kotlin` packages). The
            // `packageProviderForKotlinPackages` solves this issue by including only scopes from symbol providers which allow `kotlin`
            // packages.
            packageProviderForKotlinPackages?.doesKotlinOnlyPackageExist(fqName) == true
        } else {
            packageProvider.doesKotlinOnlyPackageExist(fqName)
        }

        // Regarding caching `hasPackage`: The static (standalone) package provider precomputes its packages, while the IDE package provider
        // caches the results itself. Hence, it's currently unnecessary to provide another layer of caching here.
        return hasPackage
    }

    override fun estimateSymbolCacheSize(): Long = classifierCache.estimatedSize

    companion object {
        fun merge(session: LLFirSession, project: Project, providers: List<LLKotlinSymbolProvider>): FirSymbolProvider? =
            if (providers.size > 1) {
                val declarationProvider = project.mergeDeclarationProviders(providers.map { it.declarationProvider })

                val packageProvider = project.mergePackageProviders(providers.map { it.packageProvider })

                val packageProviderForKotlinPackages = providers
                    .filter { it.allowKotlinPackage }
                    .takeIf { it.isNotEmpty() }
                    ?.map { it.packageProvider }
                    ?.let(project::mergePackageProviders)

                LLCombinedKotlinSymbolProvider(
                    session,
                    project,
                    providers,
                    declarationProvider,
                    packageProvider,
                    packageProviderForKotlinPackages,
                )
            } else providers.singleOrNull()
    }
}
