/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.common.lower

import ksp.org.jetbrains.kotlin.backend.common.CommonBackendContext
import ksp.org.jetbrains.kotlin.backend.common.FileLoweringPass
import ksp.org.jetbrains.kotlin.backend.common.phaser.PhaseDescription
import ksp.org.jetbrains.kotlin.config.CommonConfigurationKeys
import ksp.org.jetbrains.kotlin.ir.declarations.IrFile
import ksp.org.jetbrains.kotlin.ir.interpreter.IrInterpreter
import ksp.org.jetbrains.kotlin.ir.interpreter.IrInterpreterConfiguration
import ksp.org.jetbrains.kotlin.ir.interpreter.IrInterpreterEnvironment
import ksp.org.jetbrains.kotlin.ir.interpreter.checker.EvaluationMode
import ksp.org.jetbrains.kotlin.ir.interpreter.transformer.runConstOptimizations
import ksp.org.jetbrains.kotlin.platform.isJs
import ksp.org.jetbrains.kotlin.platform.isWasm

/**
 * Evaluates functions that are annotated with [kotlin.internal.IntrinsicConstEvaluation].
 */
@PhaseDescription(name = "ConstEvaluationLowering")
class ConstEvaluationLowering(
    val context: CommonBackendContext,
    private val suppressErrors: Boolean = context.configuration.getBoolean(CommonConfigurationKeys.IGNORE_CONST_OPTIMIZATION_ERRORS),
    configuration: IrInterpreterConfiguration = IrInterpreterConfiguration(printOnlyExceptionMessage = true),
) : FileLoweringPass {
    private val interpreter = IrInterpreter(IrInterpreterEnvironment(context.irBuiltIns, configuration), emptyMap())
    private val evaluatedConstTracker = context.configuration[CommonConfigurationKeys.EVALUATED_CONST_TRACKER]
    private val inlineConstTracker = context.configuration[CommonConfigurationKeys.INLINE_CONST_TRACKER]
    private val mode = EvaluationMode.OnlyIntrinsicConst(isFloatingPointOptimizationDisabled = configuration.platform.isJs() || configuration.platform.isWasm())

    override fun lower(irFile: IrFile) {
        irFile.runConstOptimizations(interpreter, mode, evaluatedConstTracker, inlineConstTracker, suppressErrors)
    }
}
