/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirClass
import ksp.org.jetbrains.kotlin.fir.declarations.delegateFieldsMap
import ksp.org.jetbrains.kotlin.fir.expressions.FirCall
import ksp.org.jetbrains.kotlin.fir.expressions.toReference
import ksp.org.jetbrains.kotlin.fir.references.isError
import ksp.org.jetbrains.kotlin.fir.symbols.SymbolInternals
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.fir.types.isSubtypeOf
import ksp.org.jetbrains.kotlin.fir.types.resolvedType

object FirDelegateFieldTypeMismatchChecker : FirClassChecker(MppCheckerKind.Common) {
    @SymbolInternals
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirClass) {
        for (it in declaration.superTypeRefs.indices) {
            val supertype = declaration.superTypeRefs[it]
            val field = declaration.delegateFieldsMap?.get(it)?.fir ?: continue
            val initializer = field.initializer ?: continue
            val isReportedByErrorNodeDiagnosticCollector = initializer is FirCall && initializer.toReference(context.session)?.isError() == true

            if (
                !isReportedByErrorNodeDiagnosticCollector &&
                !initializer.resolvedType.isSubtypeOf(supertype.coneType, context.session, true)
            ) {
                reporter.reportOn(
                    initializer.source,
                    FirErrors.TYPE_MISMATCH,
                    field.returnTypeRef.coneType,
                    initializer.resolvedType,
                    false
                )
            }
        }
    }
}
