/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirAnonymousFunction
import ksp.org.jetbrains.kotlin.fir.declarations.InlineStatus
import ksp.org.jetbrains.kotlin.fir.declarations.getAnnotationRetention
import ksp.org.jetbrains.kotlin.fir.declarations.toAnnotationClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnonymousFunctionExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.resolvedArgumentMapping
import ksp.org.jetbrains.kotlin.fir.expressions.unwrapArgument
import ksp.org.jetbrains.kotlin.fir.references.isError

object FirInlinedLambdaNonSourceAnnotationsChecker : FirAnonymousFunctionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirAnonymousFunction) {
        if (declaration.inlineStatus != InlineStatus.Inline && declaration.inlineStatus != InlineStatus.CrossInline) {
            return
        }

        // If the lambda belongs to a function call which is already an error, do not report non-source annotation.
        for (call in context.callsOrAssignments) {
            if (call is FirFunctionCall && call.calleeReference.isError()) {
                val mapping = call.resolvedArgumentMapping ?: continue
                for ((argument, parameter) in mapping) {
                    if ((argument.unwrapArgument() as? FirAnonymousFunctionExpression)?.anonymousFunction === declaration) {
                        return
                    }
                }
            }
        }

        for (it in declaration.annotations) {
            val annotationSymbol = it.toAnnotationClassLikeSymbol(context.session) ?: continue

            if (annotationSymbol.getAnnotationRetention(context.session) != AnnotationRetention.SOURCE) {
                reporter.reportOn(it.source, FirErrors.NON_SOURCE_ANNOTATION_ON_INLINED_LAMBDA_EXPRESSION)
            }
        }
    }
}
