/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.declaration

import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.descriptors.Modality
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirBasicDeclarationChecker
import ksp.org.jetbrains.kotlin.fir.resolve.getContainingClassSymbol
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.declarations.*
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInlineOrValue
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isOverridable
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isOverride
import ksp.org.jetbrains.kotlin.fir.declarations.utils.modality
import ksp.org.jetbrains.kotlin.fir.expressions.FirLiteralExpression
import ksp.org.jetbrains.kotlin.fir.java.findJvmNameAnnotation
import ksp.org.jetbrains.kotlin.fir.resolve.getContainingClass
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.name.StandardClassIds

object FirJvmNameChecker : FirBasicDeclarationChecker(MppCheckerKind.Common) {

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirDeclaration) {
        val jvmName = declaration.findJvmNameAnnotation() ?: return
        val name = jvmName.findArgumentByName(StandardNames.NAME) ?: return

        if (name.resolvedType != context.session.builtinTypes.stringType.coneType) {
            return
        }

        val value = (name as? FirLiteralExpression)?.value as? String ?: return

        if (!Name.isValidIdentifier(value)) {
            reporter.reportOn(jvmName.source, FirJvmErrors.ILLEGAL_JVM_NAME)
        }

        if (declaration is FirFunction && !context.isRenamableFunction(declaration)) {
            reporter.reportOn(jvmName.source, FirJvmErrors.INAPPLICABLE_JVM_NAME)
        } else if (declaration is FirCallableDeclaration) {
            val containingClass = declaration.getContainingClass()

            if (
                declaration.isOverride ||
                containingClass != null && containingClass.modality != Modality.FINAL && declaration.isOverridable ||
                containingClass?.isValueClassThatRequiresMangling() == true
            ) {
                reporter.reportOn(jvmName.source, FirJvmErrors.INAPPLICABLE_JVM_NAME)
            }
        }
    }

    private fun CheckerContext.isRenamableFunction(function: FirFunction): Boolean {
        val containingClass = function.getContainingClassSymbol()
        return containingClass != null || !function.symbol.callableId.isLocal
    }

    private fun FirRegularClass.isValueClassThatRequiresMangling(): Boolean {
        // value classes have inline modifiers in FIR
        return isInlineOrValue && name != StandardClassIds.Result.shortClassName
    }
}
