/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.descriptors.annotations.KotlinTarget
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirAnnotationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getActualTargetList
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.declarations.InlineStatus.*
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.resolve.fqName
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirAnonymousFunctionSymbol
import ksp.org.jetbrains.kotlin.name.JvmStandardClassIds.JVM_SERIALIZABLE_LAMBDA_ANNOTATION_FQ_NAME

object FirJvmSerializableLambdaChecker : FirAnnotationChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirAnnotation) {
        if (expression.fqName(context.session) == JVM_SERIALIZABLE_LAMBDA_ANNOTATION_FQ_NAME) {
            val declaration = context.containingDeclarations.last()
            if (declaration !is FirAnonymousFunctionSymbol) {
                val actualTargets = getActualTargetList(declaration)
                val targetDescription = actualTargets.defaultTargets.firstOrNull()?.description ?: "unidentified target"
                reporter.reportOn(
                    expression.source,
                    FirErrors.WRONG_ANNOTATION_TARGET,
                    targetDescription,
                    listOf(KotlinTarget.LAMBDA_EXPRESSION)
                )
            } else {
                val diagnostic = when (declaration.inlineStatus) {
                    Inline, CrossInline -> FirJvmErrors.JVM_SERIALIZABLE_LAMBDA_ON_INLINED_FUNCTION_LITERALS
                    NoInline, Unknown -> return
                }
                reporter.reportOn(expression.source, diagnostic)
            }
        }
    }
}
