/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.wasm.checkers

import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.fir.declarations.FirSimpleFunction
import ksp.org.jetbrains.kotlin.fir.expressions.FirBlock
import ksp.org.jetbrains.kotlin.fir.expressions.FirExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.FirReturnExpression
import ksp.org.jetbrains.kotlin.fir.expressions.impl.FirSingleExpressionBlock
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.SymbolInternals
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.name.WebCommonStandardClassIds


@OptIn(SymbolInternals::class)
fun FirNamedFunctionSymbol.hasValidJsCodeBody(): Boolean = fir.hasValidJsCodeBody()

fun FirSimpleFunction.hasValidJsCodeBody(): Boolean =
    body?.isValidJsCodeBody() == true

@OptIn(SymbolInternals::class)
fun FirPropertySymbol.hasValidJsCodeBody(): Boolean = fir.hasValidJsCodeBody()

fun FirProperty.hasValidJsCodeBody(): Boolean =
    this.initializer?.isJsCodeCall() == true

private fun FirBlock.isValidJsCodeBody(): Boolean {
    val singleStatement = statements.singleOrNull()
        ?: return false

    return when {
        singleStatement is FirFunctionCall ->
            singleStatement.isJsCodeCall()

        singleStatement is FirReturnExpression && this is FirSingleExpressionBlock ->
            singleStatement.result.isJsCodeCall()

        else ->
            false
    }
}

private fun FirExpression.isJsCodeCall(): Boolean {
    if (this !is FirFunctionCall)
        return false

    val symbol = calleeReference.toResolvedCallableSymbol()
        ?: return false

    return symbol.callableId == WebCommonStandardClassIds.Callables.Js
}
