/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.serialization

import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.FirClass
import ksp.org.jetbrains.kotlin.fir.declarations.FirClassLikeDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirFile
import ksp.org.jetbrains.kotlin.fir.packageFqName
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassSymbol
import ksp.org.jetbrains.kotlin.library.metadata.buildKlibPackageFragment
import ksp.org.jetbrains.kotlin.metadata.ProtoBuf
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.serialization.SerializableStringTable

// TODO: handle incremental/monolothic (see klib serializer) - maybe externally
fun serializeSingleFirFile(
    file: FirFile, session: FirSession, scopeSession: ScopeSession,
    actualizedExpectDeclarations: Set<FirDeclaration>?,
    serializerExtension: FirKLibSerializerExtension,
    languageVersionSettings: LanguageVersionSettings,
    produceHeaderKlib: Boolean = false,
): ProtoBuf.PackageFragment {
    val approximator = TypeApproximatorForMetadataSerializer(session)

    val packageSerializer = FirElementSerializer.createTopLevel(
        session, scopeSession, serializerExtension,
        approximator,
        languageVersionSettings,
        produceHeaderKlib
    )
    val packageProto = packageSerializer.packagePartProto(file, actualizedExpectDeclarations).build()

    val classesProto = mutableListOf<Pair<ProtoBuf.Class, Int>>()

    fun FirClass.makeClassProtoWithNested() {
        if (!isNotExpectOrShouldBeSerialized(actualizedExpectDeclarations) ||
            !isNotPrivateOrShouldBeSerialized(produceHeaderKlib)
        ) {
            return
        }

        val classSerializer = FirElementSerializer.create(
            session, scopeSession, klass = this, serializerExtension, parentSerializer = null,
            approximator, languageVersionSettings, produceHeaderKlib
        )
        val index = classSerializer.stringTable.getFqNameIndex(this)

        classesProto += classSerializer.classProto(this, file).build() to index

        for (nestedClassifierSymbol in classSerializer.computeNestedClassifiersForClass(symbol)) {
            (nestedClassifierSymbol as? FirClassSymbol<*>)?.fir?.makeClassProtoWithNested()
        }
    }

    serializerExtension.processFile(file) {
        for (declaration in file.declarations) {
            (declaration as? FirClass)?.makeClassProtoWithNested()
        }
    }

    return buildKlibPackageFragment(
        packageProto,
        classesProto,
        file.packageFqName,
        isEmpty = packageProto.functionList.isEmpty() &&
                packageProto.propertyList.isEmpty() &&
                packageProto.typeAliasList.isEmpty() &&
                classesProto.isEmpty(),
        serializerExtension.stringTable as SerializableStringTable
    )
}

class FirElementAwareSerializableStringTable : FirElementAwareStringTable, SerializableStringTable() {
    override fun getLocalClassLikeDeclarationIdReplacement(declaration: FirClassLikeDeclaration): ClassId = StandardClassIds.Any
}
