/*
 * Copyright 2000-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.resolve

import ksp.org.jetbrains.kotlin.builtins.KotlinBuiltIns
import ksp.org.jetbrains.kotlin.descriptors.CallableDescriptor
import ksp.org.jetbrains.kotlin.descriptors.FunctionDescriptor
import ksp.org.jetbrains.kotlin.descriptors.VariableDescriptorWithAccessors
import ksp.org.jetbrains.kotlin.resolve.calls.components.*
import ksp.org.jetbrains.kotlin.resolve.calls.components.candidate.ResolutionCandidate
import ksp.org.jetbrains.kotlin.resolve.calls.inference.BuilderInferenceSession
import ksp.org.jetbrains.kotlin.resolve.calls.inference.ConstraintSystemBuilder
import ksp.org.jetbrains.kotlin.resolve.calls.inference.NewConstraintSystem
import ksp.org.jetbrains.kotlin.resolve.calls.inference.components.ConstraintSystemCompletionMode
import ksp.org.jetbrains.kotlin.resolve.calls.inference.components.KotlinConstraintSystemCompleter
import ksp.org.jetbrains.kotlin.resolve.calls.inference.model.ConstraintStorage
import ksp.org.jetbrains.kotlin.resolve.calls.inference.model.DelegatedPropertyConstraintPositionImpl
import ksp.org.jetbrains.kotlin.resolve.calls.model.*
import ksp.org.jetbrains.kotlin.resolve.calls.tower.StubTypesBasedInferenceSession
import ksp.org.jetbrains.kotlin.resolve.calls.tower.PSICallResolver
import ksp.org.jetbrains.kotlin.resolve.calls.tower.PSIPartialCallInfo
import ksp.org.jetbrains.kotlin.types.error.ErrorUtils
import ksp.org.jetbrains.kotlin.types.TypeConstructor
import ksp.org.jetbrains.kotlin.types.UnwrappedType
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

class DelegateInferenceSession(
    val variableDescriptor: VariableDescriptorWithAccessors,
    val expectedType: UnwrappedType?,
    psiCallResolver: PSICallResolver,
    postponedArgumentsAnalyzer: PostponedArgumentsAnalyzer,
    kotlinConstraintSystemCompleter: KotlinConstraintSystemCompleter,
    callComponents: KotlinCallComponents,
    builtIns: KotlinBuiltIns,
    override val parentSession: InferenceSession?
) : StubTypesBasedInferenceSession<FunctionDescriptor>(
    psiCallResolver, postponedArgumentsAnalyzer, kotlinConstraintSystemCompleter, callComponents, builtIns
) {
    init {
        if (parentSession is StubTypesBasedInferenceSession<*>) {
            parentSession.addNestedInferenceSession(this)
        }
    }

    fun getNestedBuilderInferenceSessions(): List<BuilderInferenceSession> {
        val builderInferenceSessions = nestedInferenceSessions.filterIsInstance<BuilderInferenceSession>()
        val delegatedPropertyInferenceSessions = nestedInferenceSessions.filterIsInstance<DelegateInferenceSession>()

        return builderInferenceSessions + delegatedPropertyInferenceSessions.map { it.getNestedBuilderInferenceSessions() }.flatten()
    }

    override fun prepareForCompletion(commonSystem: NewConstraintSystem, resolvedCallsInfo: List<PSIPartialCallInfo>) {
        val csBuilder = commonSystem.getBuilder()
        for (callInfo in resolvedCallsInfo) {
            val resultAtom = callInfo.callResolutionResult.resultCallAtom
            when (resultAtom.candidateDescriptor.name) {
                OperatorNameConventions.GET_VALUE -> resultAtom.addConstraintsForGetValueMethod(csBuilder)
                OperatorNameConventions.SET_VALUE -> resultAtom.addConstraintsForSetValueMethod(csBuilder)
            }
        }
    }

    private fun ResolvedCallAtom.addConstraintForThis(descriptor: CallableDescriptor, commonSystem: ConstraintSystemBuilder) {
        val typeOfThis = variableDescriptor.extensionReceiverParameter?.type
            ?: variableDescriptor.dispatchReceiverParameter?.type
            ?: builtIns.nullableNothingType

        val valueParameterForThis = descriptor.valueParameters.getOrNull(0) ?: return
        val substitutedType = freshVariablesSubstitutor.safeSubstitute(valueParameterForThis.type.unwrap())
        commonSystem.addSubtypeConstraint(typeOfThis.unwrap(), substitutedType, DelegatedPropertyConstraintPositionImpl(atom))
    }

    private fun ResolvedCallAtom.addConstraintsForGetValueMethod(commonSystem: ConstraintSystemBuilder) {
        if (expectedType != null) {
            val unsubstitutedReturnType = candidateDescriptor.returnType?.unwrap() ?: return
            val substitutedReturnType = freshVariablesSubstitutor.safeSubstitute(unsubstitutedReturnType)

            commonSystem.addSubtypeConstraint(substitutedReturnType, expectedType, DelegatedPropertyConstraintPositionImpl(atom))
        }

        addConstraintForThis(candidateDescriptor, commonSystem)
    }

    private fun ResolvedCallAtom.addConstraintsForSetValueMethod(commonSystem: ConstraintSystemBuilder) {
        if (expectedType != null) {
            val unsubstitutedParameterType = candidateDescriptor.valueParameters.getOrNull(2)?.type?.unwrap() ?: return
            val substitutedParameterType = freshVariablesSubstitutor.safeSubstitute(unsubstitutedParameterType)

            commonSystem.addSubtypeConstraint(expectedType, substitutedParameterType, DelegatedPropertyConstraintPositionImpl(atom))
        }

        addConstraintForThis(candidateDescriptor, commonSystem)
    }

    override fun inferPostponedVariables(
        lambda: ResolvedLambdaAtom,
        constraintSystemBuilder: ConstraintSystemBuilder,
        completionMode: ConstraintSystemCompletionMode,
        diagnosticsHolder: KotlinDiagnosticsHolder
    ): Map<TypeConstructor, UnwrappedType> = emptyMap()

    override fun initializeLambda(lambda: ResolvedLambdaAtom) {}

    override fun writeOnlyStubs(callInfo: SingleCallResolutionResult): Boolean = false

    override fun shouldCompleteResolvedSubAtomsOf(resolvedCallAtom: ResolvedCallAtom) = true
}

class InferenceSessionForExistingCandidates(
    private val resolveReceiverIndependently: Boolean,
    override val parentSession: InferenceSession?
) : InferenceSession {
    override fun shouldRunCompletion(candidate: ResolutionCandidate): Boolean {
        return !ErrorUtils.isError(candidate.resolvedCall.candidateDescriptor)
    }

    override fun addPartialCallInfo(callInfo: PartialCallInfo) {}
    override fun addCompletedCallInfo(callInfo: CompletedCallInfo) {}
    override fun addErrorCallInfo(callInfo: ErrorCallInfo) {}

    override fun currentConstraintSystem(): ConstraintStorage = ConstraintStorage.Empty
    override fun inferPostponedVariables(
        lambda: ResolvedLambdaAtom,
        constraintSystemBuilder: ConstraintSystemBuilder,
        completionMode: ConstraintSystemCompletionMode,
        diagnosticsHolder: KotlinDiagnosticsHolder
    ): Map<TypeConstructor, UnwrappedType> = emptyMap()

    override fun writeOnlyStubs(callInfo: SingleCallResolutionResult): Boolean = false
    override fun callCompleted(resolvedAtom: ResolvedAtom): Boolean = false
    override fun shouldCompleteResolvedSubAtomsOf(resolvedCallAtom: ResolvedCallAtom): Boolean {
        return !ErrorUtils.isError(resolvedCallAtom.candidateDescriptor)
    }

    override fun computeCompletionMode(
        candidate: ResolutionCandidate
    ): ConstraintSystemCompletionMode? = null

    override fun resolveReceiverIndependently(): Boolean = resolveReceiverIndependently

    override fun initializeLambda(lambda: ResolvedLambdaAtom) {}
}
