/*
 * Copyright 2010-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.resolve.checkers

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.descriptors.ClassDescriptor
import ksp.org.jetbrains.kotlin.descriptors.DeclarationDescriptor
import ksp.org.jetbrains.kotlin.descriptors.annotations.AnnotationDescriptor
import ksp.org.jetbrains.kotlin.descriptors.annotations.KotlinRetention
import ksp.org.jetbrains.kotlin.descriptors.annotations.KotlinTarget
import ksp.org.jetbrains.kotlin.diagnostics.Errors
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtDeclaration
import ksp.org.jetbrains.kotlin.resolve.AnnotationChecker
import ksp.org.jetbrains.kotlin.resolve.DescriptorToSourceUtils
import ksp.org.jetbrains.kotlin.resolve.DescriptorUtils
import ksp.org.jetbrains.kotlin.resolve.descriptorUtil.getAnnotationRetention

class AnnotationClassTargetAndRetentionChecker : DeclarationChecker {
    override fun check(declaration: KtDeclaration, descriptor: DeclarationDescriptor, context: DeclarationCheckerContext) {
        if (descriptor !is ClassDescriptor) return
        if (declaration !is KtClassOrObject) return
        if (!DescriptorUtils.isAnnotationClass(descriptor)) return

        val targets = AnnotationChecker.applicableTargetSetFromTargetAnnotationOrNull(descriptor) ?: return
        val retention = descriptor.getAnnotationRetention() ?: KotlinRetention.RUNTIME

        if (targets.contains(KotlinTarget.EXPRESSION) && retention != KotlinRetention.SOURCE) {
            val retentionAnnotation = descriptor.annotations.findAnnotation(StandardNames.FqNames.retention)
            val targetAnnotation = descriptor.annotations.findAnnotation(StandardNames.FqNames.target)

            context.trace.report(
                Errors.RESTRICTED_RETENTION_FOR_EXPRESSION_ANNOTATION.on(
                    context.languageVersionSettings,
                    retentionAnnotation?.psi ?: targetAnnotation?.psi ?: declaration
                )
            )
        }
    }

    private val AnnotationDescriptor.psi: PsiElement?
        get() = DescriptorToSourceUtils.getSourceFromAnnotation(this)
}
