/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */
package ksp.org.jetbrains.kotlin.analysis.decompiler.konan

import ksp.com.intellij.openapi.fileTypes.FileType
import ksp.com.intellij.openapi.project.DefaultProjectFactory
import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.com.intellij.psi.PsiFileFactory
import ksp.com.intellij.psi.SingleRootFileViewProvider
import ksp.com.intellij.psi.impl.PsiFileFactoryImpl
import ksp.com.intellij.psi.stubs.PsiFileStub
import ksp.com.intellij.testFramework.LightVirtualFile
import ksp.com.intellij.util.indexing.FileContent
import ksp.org.jetbrains.kotlin.analysis.decompiler.psi.text.defaultDecompilerRendererOptions
import ksp.org.jetbrains.kotlin.idea.KotlinFileType
import ksp.org.jetbrains.kotlin.idea.KotlinLanguage
import ksp.org.jetbrains.kotlin.psi.stubs.elements.KtFileElementType
import ksp.org.jetbrains.kotlin.renderer.DescriptorRenderer
import ksp.org.jetbrains.kotlin.serialization.SerializerExtensionProtocol
import ksp.org.jetbrains.kotlin.serialization.js.DynamicTypeDeserializer

internal open class Fe10KlibMetadataStubBuilder(
    version: Int,
    fileType: FileType,
    private val serializerProtocol: () -> SerializerExtensionProtocol,
    readFile: (VirtualFile) -> FileWithMetadata?
) : KlibMetadataStubBuilder(version, fileType, readFile) {
    private val renderer: DescriptorRenderer = DescriptorRenderer.withOptions { defaultDecompilerRendererOptions() }

    override fun buildMetadataFileStub(fileWithMetadata: FileWithMetadata.Compatible, fileContent: FileContent): PsiFileStub<*> {
        val ktFileText = decompiledText(
            fileWithMetadata,
            fileContent.file,
            serializerProtocol(),
            readFile,
            DynamicTypeDeserializer,
            renderer
        )

        val project = fileContent.project ?: DefaultProjectFactory.getInstance().defaultProject
        return createFileStub(project, ktFileText.text)
    }

    private fun createFileStub(project: Project, text: String): PsiFileStub<*> {
        val virtualFile = LightVirtualFile("dummy.kt", KotlinFileType.INSTANCE, text)
        virtualFile.language = KotlinLanguage.INSTANCE
        SingleRootFileViewProvider.doNotCheckFileSizeLimit(virtualFile)

        val psiFileFactory = PsiFileFactory.getInstance(project) as PsiFileFactoryImpl
        val file = psiFileFactory.trySetupPsiForFile(virtualFile, KotlinLanguage.INSTANCE, false, false)!!
        return KtFileElementType.builder.buildStubTree(file) as PsiFileStub<*>
    }
}
