/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.descriptors.Visibilities
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.findClosest
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirBasicExpressionChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.declarations.hasAnnotation
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isCompanion
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isConst
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.expressions.*
import ksp.org.jetbrains.kotlin.fir.isEnabled
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.defaultType
import ksp.org.jetbrains.kotlin.fir.resolve.getContainingDeclaration
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.typeContext
import ksp.org.jetbrains.kotlin.load.java.JvmAbi
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.types.AbstractTypeChecker

// TODO: consider what to do with it
object FirJvmProtectedInSuperClassCompanionCallChecker : FirBasicExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirStatement) {
        val dispatchReceiver = when (expression) {
            is FirQualifiedAccessExpression -> expression.dispatchReceiver
            is FirVariableAssignment -> expression.dispatchReceiver
            else -> null
        } ?: return

        val dispatchClassSymbol = dispatchReceiver.resolvedType.toRegularClassSymbol(context.session) ?: return
        val calleeReference = expression.toReference(context.session)
        val resolvedSymbol = calleeReference?.toResolvedCallableSymbol() ?: return

        if (resolvedSymbol is FirPropertySymbol &&
            LanguageFeature.AllowAccessToProtectedFieldFromSuperCompanion.isEnabled()
        ) {
            if (resolvedSymbol.isConst) return

            val backingField = resolvedSymbol.backingFieldSymbol
            if (backingField != null && backingField.hasAnnotation(JvmAbi.JVM_FIELD_ANNOTATION_CLASS_ID, context.session)) return
        }

        val visibility = if (resolvedSymbol is FirPropertySymbol) {
            if (expression is FirVariableAssignment)
                resolvedSymbol.setterSymbol?.visibility ?: resolvedSymbol.visibility
            else
                resolvedSymbol.getterSymbol?.visibility ?: resolvedSymbol.visibility
        } else {
            resolvedSymbol.visibility
        }
        if (visibility != Visibilities.Protected) return
        if (resolvedSymbol.hasAnnotation(StandardClassIds.Annotations.jvmStatic, context.session)) return
        if (!dispatchClassSymbol.isCompanion) return
        val companionContainingClassSymbol =
            dispatchClassSymbol.getContainingDeclaration(context.session) as? FirRegularClassSymbol ?: return

        // Called from within a derived class
        val companionContainingType = companionContainingClassSymbol.defaultType()
        if (context.findClosest<FirClassSymbol<*>> {
                AbstractTypeChecker.isSubtypeOf(context.session.typeContext, it.defaultType(), companionContainingType)
            } == null
        ) {
            return
        }

        // Called not within the same companion object or its owner class
        if (context.findClosest<FirClassSymbol<*>> {
                it == dispatchClassSymbol || it == companionContainingClassSymbol
            } == null
        ) {
            reporter.reportOn(calleeReference.source, FirJvmErrors.SUBCLASS_CANT_CALL_COMPANION_PROTECTED_NON_STATIC)
        }
    }
}
