/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.deserialization

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isStatic
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeSimpleDiagnostic
import ksp.org.jetbrains.kotlin.fir.diagnostics.DiagnosticKind
import ksp.org.jetbrains.kotlin.fir.expressions.FirEnumEntryDeserializedAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirPropertyAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.expressions.builder.buildPropertyAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.builder.buildResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.references.builder.buildErrorNamedReference
import ksp.org.jetbrains.kotlin.fir.references.builder.buildResolvedNamedReference
import ksp.org.jetbrains.kotlin.fir.resolve.providers.getClassDeclaredPropertySymbols
import ksp.org.jetbrains.kotlin.fir.resolve.toSymbol
import ksp.org.jetbrains.kotlin.fir.types.constructClassType
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.fir.types.toLookupTag
import ksp.org.jetbrains.kotlin.name.ClassId

fun FirEnumEntryDeserializedAccessExpression.toQualifiedPropertyAccessExpression(session: FirSession): FirPropertyAccessExpression =
    buildPropertyAccessExpression {
        val entryPropertySymbol = session.getClassDeclaredPropertySymbols(
            enumClassId, enumEntryName,
        ).firstOrNull { it.isStatic }

        calleeReference = when {
            entryPropertySymbol != null -> {
                buildResolvedNamedReference {
                    this.name = enumEntryName
                    resolvedSymbol = entryPropertySymbol
                }
            }
            else -> {
                buildErrorNamedReference {
                    diagnostic = ConeSimpleDiagnostic(
                        "Strange deserialized enum value: $enumClassId.$enumEntryName",
                        DiagnosticKind.Java,
                    )
                    name = enumEntryName
                }
            }
        }

        val receiver = enumClassId.toResolvedQualifier(session)
        coneTypeOrNull = receiver.resolvedType
        dispatchReceiver = receiver
        explicitReceiver = receiver
    }

fun ClassId.toResolvedQualifier(session: FirSession): FirResolvedQualifier {
    val lookupTag = toLookupTag()

    return buildResolvedQualifier {
        coneTypeOrNull = lookupTag.constructClassType()
        packageFqName = this@toResolvedQualifier.packageFqName
        relativeClassFqName = relativeClassName
        symbol = lookupTag.toSymbol(session)
        resolvedToCompanionObject = false
    }
}
